(ns parts.components.datomic
  (:require
   [com.stuartsierra.component :as c]
   [datomic.api :as dtm]))

;; -----------------------------------------------------
;; datomic blueprint
;; -----------------------------------------------------

(defrecord TemporaryDatomicBlueprint [uri started?]
  c/Lifecycle
  (start [this]
    (if started?
      this
      (do (dtm/create-database uri)
          (assoc this :started? true))))
  (stop [this]
    (if-not started?
      this
      (do (dtm/delete-database uri)
          (assoc this :started? false)))))

(defn- make-temporary-datomic-blueprint
  [config]
  (-> config
      (select-keys [:uri])
      (assoc :started? false)
      (map->TemporaryDatomicBlueprint)))

(defrecord DurableDatomicBlueprint [uri started?]
  c/Lifecycle
  (start [this]
    (if started?
      this
      (do (dtm/create-database uri)
          (assoc this :started? true))))
  (stop [this]
    (if-not started?
      this
      (assoc this :started? false))))

(defn- make-durable-datomic-blueprint
  [config]
  (-> config
      (select-keys [:uri])
      (assoc :started? false)
      (map->DurableDatomicBlueprint)))

(defn make-datomic-blueprint
  [{:keys [kind] :as config}]
  (case kind
    :durable   (make-durable-datomic-blueprint config)
    :temporary (make-temporary-datomic-blueprint config)))

;; -----------------------------------------------------
;; datomic connection
;; -----------------------------------------------------

(defrecord DatomicConnection [datomic-blueprint connection]
  c/Lifecycle
  (start [this]
    (if (some? connection)
      this
      (let [uri (:uri datomic-blueprint)]
        (assoc this :connection (dtm/connect uri)))))
  (stop [this]
    (if (nil? connection)
      this
      (do (dtm/release connection)
          (assoc this :connection nil)))))

(defn make-datomic-connection
  []
  (c/using
   (map->DatomicConnection {})
   [:datomic-blueprint]))
