(ns parts.components.bidi
  (:require
   [clojure.spec.alpha :as s]
   [com.stuartsierra.component :as c]
   [bidi.bidi :as b]
   [bidi.ring :refer [make-handler]]
   [parts.components.ring :as cptrng]))

;; -----------------------------------------------------
;; route collector spec
;; -----------------------------------------------------

(s/def ::prefix
  string?)

;; -----------------------------------------------------
;; ring router spec
;; -----------------------------------------------------

(s/def ::not-found-handler
  fn?)

;; -----------------------------------------------------
;; ring endpoint spec
;; -----------------------------------------------------

(s/def ::routes
  (s/tuple (s/or :path               string?
                 :parameterized-path (s/coll-of (s/or :part  string?
                                                      :param keyword?)))
           (s/or :endpoint fn?
                 :routes   (s/coll-of ::routes))))

;; -----------------------------------------------------
;; route collector
;; -----------------------------------------------------

(defrecord RouteCollector [prefix]
  b/RouteProvider
  (routes [this]
    (if-let [routes (not-empty
                     (into []
                           (comp
                            (map val)
                            (filter #(satisfies? b/RouteProvider %))
                            (map b/routes))
                           this))]
      [prefix routes]
      (throw (ex-info "No routes collected" {})))))

(defn make-route-collector
  [prefix]
  (->RouteCollector (s/assert ::prefix prefix)))

;; -----------------------------------------------------
;; router
;; -----------------------------------------------------

(defrecord Router [route-collector not-found-handler]
  b/RouteProvider
  (routes [this]
    (b/routes route-collector))

  cptrng/IRequestHandler
  (-request-handler [this]
    (let [routes (b/routes route-collector)]
      (some-fn (make-handler routes) not-found-handler))))

(defn make-router
  [not-found-handler]
  (-> {:not-found-handler (s/assert ::not-found-handler not-found-handler)}
      (map->Router)
      (c/using [:route-collector])))

;; -----------------------------------------------------
;; route endpoint
;; -----------------------------------------------------

(defrecord RouteEndpoint [routes]
  b/RouteProvider
  (routes [this]
    routes))

(defn make-ring-endpoint
  [routes]
  (->RouteEndpoint (s/assert ::routes routes)))
