(ns parts.components.async
  (:require
   [com.stuartsierra.component :as c]
   #?@(:clj  [[clojure.spec.alpha :as s]
              [clojure.core.async :as a :refer [go-loop]]]
       :cljs [[cljs.spec.alpha :as s]
              [cljs.core.async :as a]]))
  #?(:cljs
     (:require-macros
      [cljs.core.async.macros :refer [go-loop]])))

;; -----------------------------------------------------
;; channel pipeliner spec
;; -----------------------------------------------------

(s/def ::kind
  #{:normal :async #?(:clj :blocking)})

(s/def ::parallelism
  pos-int?)

(s/def ::updater-fn
  fn?)

(s/def ::close-both?
  boolean?)

(s/def ::ex-handler
  fn?)

(s/def ::channel-pipeliner-config
  (s/keys :req-un [::kind ::updater-fn ::ex-handler]
          :opt-un [::parallelism ::close-both?]))

;; -----------------------------------------------------
;; protocols
;; -----------------------------------------------------

(defprotocol ISource
  (source-chan [this]))

(defprotocol ISink
  (sink-chan [this]))

;; -----------------------------------------------------
;; channel
;; -----------------------------------------------------

(defrecord Channel [buffer chan]
  c/Lifecycle
  (start [this]
    (if (some? chan)
      this
      (let [{:keys [fixed sliding dropping]} buffer]
        (assoc this :chan (cond
                            (pos-int? buffer)
                            (a/chan buffer)

                            (pos-int? fixed)
                            (a/chan fixed)

                            (pos-int? sliding)
                            (a/chan (a/sliding-buffer sliding))

                            (pos-int? dropping)
                            (a/chan (a/dropping-buffer dropping))

                            :else (a/chan))))))
  (stop [this]
    (if (nil? chan)
      this
      (do (a/close! chan)
          (assoc this :chan nil)))))

(defn make-channel
  [config]
  (-> config
      (select-keys [:buffer])
      (map->Channel)))

;; -----------------------------------------------------
;; channel pipeliner
;; -----------------------------------------------------

(defn- pipeline!
  [kind parallelism to-chan updater from-chan close-both? ex-handler]
  (case kind
    :normal
    (a/pipeline parallelism
                to-chan
                updater
                from-chan
                close-both?
                ex-handler)

    :async
    (a/pipeline-async parallelism
                      to-chan
                      updater
                      from-chan
                      close-both?)

    #?@(:clj [:blocking
              (a/pipeline-blocking parallelism
                                   to-chan
                                   updater
                                   from-chan
                                   close-both?
                                   ex-handler)])

    (pipeline! :normal
               parallelism
               to-chan
               updater
               from-chan
               close-both?
               ex-handler)))

(defrecord ChannelPipeliner [kind
                             parallelism
                             to
                             updater-fn
                             from
                             close-both?
                             ex-handler
                             started?]
  c/Lifecycle
  (start [this]
    (if started?
      this
      (let [parallelism (or parallelism 1)
            to-chan     (sink-chan to)
            updater     (updater-fn this)
            from-chan   (source-chan from)
            close-both? (if (nil? close-both?) true close-both?)]
        (pipeline! kind
                   parallelism
                   to-chan
                   updater
                   from-chan
                   close-both?
                   ex-handler)
        (assoc this :started? true))))
  (stop [this]
    (if-not started?
      this
      (assoc this :started? false))))

(defn make-channel-pipeliner
  [config]
  (-> (s/assert ::channel-pipeliner-config config)
      (select-keys [:kind
                    :parallelism
                    :updater-fn
                    :from
                    :close-both?
                    :ex-handler])
      (assoc :started? false)
      (map->ChannelPipeliner)
      (c/using [:to :from])))

;; -----------------------------------------------------
;; channel listener
;; -----------------------------------------------------

(defn- collect-source-chans
  [this]
  (if-let [source-chans (not-empty (into []
                                         (comp
                                          (map val)
                                          (filter #(satisfies? ISource %))
                                          (map source-chan))
                                         this))]
    source-chans
    (throw (ex-info "No source chan collected" {}))))

(defrecord ChannelListener [callback stop-chan]
  c/Lifecycle
  (start [this]
    (if (some? stop-chan)
      this
      (let [source-chans (collect-source-chans this)
            stop-chan    (a/chan)]
        (go-loop []
          (let [[item chan]
                (a/alts! (conj source-chans stop-chan) :priority true)

                stop? (or (= stop-chan chan) (nil? item))]
            (when-not stop?
              (callback this item)
              (recur))))
        (assoc this :stop-chan stop-chan))))
  (stop [this]
    (if (nil? stop-chan)
      this
      (do (a/close! stop-chan)
          (assoc this :stop-chan nil)))))

(defn make-channel-listener
  []
  (map->ChannelListener {}))
