(ns fdb-clj.core
  (:require [fdb-clj.tuple :as t])
  (:import [com.apple.foundationdb FDB Database Range]
           [com.apple.foundationdb.tuple Tuple]))

(defmacro tr!
  "Transaction macro to perform actions. Always use tr for actions inside
  each action since the transaction variable is bound to tr in the functions.
  ```
  (let [fd    (select-api-version 520)
        key   \"foo\"
        value \"bar\"]
  (with-open [db (open fd)]
     (tr! db
          (set-val tr key value)
          (get-val tr key))))
  ```
  "
  [db & actions]
  `(.run ~db
         (reify
           java.util.function.Function
           (apply [this tr]
             ~@actions))))

(defn open
  "
  Initializes networking, connects with the default fdb.cluster file,
  and opens the database.
  "
  [^FDB db]
  (if (delay? db)
    (.open @db)
    (.open db)))

(defn key->tuple
  "
  Return tuple encoding for the given key
  "
  [key]
  (let [key (if (sequential? key) key [key])]
    (Tuple/from (to-array key))))

(defn key->packed-tuple
  "
  Pack the key with respect to Tuple encoding
  "
  [key]
  (.pack (key->tuple key)))

(def ^:dynamic *directory* nil)

(defn set-val
  "Set a value for the key. Accepts the below :
  :subspace - Subspace to be prefixed
  ```
  (let [fd    (select-api-version 520)
        key   \"foo\"
        value \"bar\"]
  (with-open [db (open fd)]
     (tr! db
          (set-val tr key value))))
  ```
  "
  [tr key val & {:keys [directory] :or {directory *directory*}}]
  (let [tuple (Tuple/fromList key)
        packed (if (nil? directory) (.pack tuple) (.pack @directory tuple))
        value-bs (:packed (t/make val))]
    (.set tr packed value-bs)))

(defn get-val
  "Get the value for the given key. Accepts the below :
  :subspace - Subspace to be prefixed
  :coll     - Boolean to indicate if the value needs to be deserialized as collection
  ```
  (let [fd  (select-api-version 520)
        key \"foo\"]
  (with-open [db (open fd)]
     (tr! db
          (get-val tr key))))
  (let [fd    (select-api-version 520)
       key   \"foo\"
       value [1 2 3]]
  (with-open [db (open fd)]
    (tr! db
         (set-val tr key value)
         (get-val tr key) ;; 1
         (get-val tr key :coll true)))) ;; [1 2 3]
  ```
  "
  [tr key & {:keys [directory] :or {directory *directory*}}]
  ;(println "The key inside get-val is --> " key)
  ;(println "The key inside tr is --> " tr)
  ;(println "The type of dir is is --> " (type directory))
  (let [tuple (Tuple/fromList key)
  ;      _     (println "created tuple")
  ;      _     (println "The dir is --" directory)
        packed (if (nil? directory) (.pack tuple) (.pack @directory tuple))]
  ;      _ (println "packed!!!!!")]
    (.get tr packed)))

(defn get-range
  [tr {start-path :path start-key :key} {end-path :path end-key :key}]
  ;(let [;start (tuple/map->tuple {:path start-path :key start-key})
  ;      ;end   (.pack (tuple/map->tuple {:path end-path :key end-key}) start-path)]
  ;      ;r     (Range. start end)]
  (println "Start-path --> " start-path)
    (.iterator (.getRange tr (.range (Tuple/fromList start-path)))))

(defn make-range
  [tr start & {:keys [directory] :or {directory *directory*}}]
  (let [tuple (Tuple/fromList start)
        rng   (if (nil? directory) (.range tuple) (.range @directory tuple))]
    (.iterator (.getRange tr rng))))

(defn watch!
  [tr key & {:keys [directory] :or {directory *directory*}}]
  (let [key-tuple (t/make key)
        packed    (if (nil? directory) (:packed key-tuple) (.pack @directory (:tuple key-tuple)))]
    (.watch tr packed)))

(defn delete-key
  [tr key & {:keys [directory] :or {directory *directory*}}]
  (cond
    (true? (instance? com.apple.foundationdb.Range key)) (.clear tr key)
    (true? (bytes? key))                                 (.clear tr key)
    :else (let [ts     (if (sequential? key) key (conj [] key))
                tuple  (Tuple/fromList ts)
                packed (if (nil? directory) (.pack tuple) (.pack @directory tuple))]
            (.clear tr packed))))

