(ns utility-belt.config
  "Config reading/loading helpers.
   Adds extensions to aero and default config reader."
  (:require
    [aero.core :as aero]
    [clojure.string :as string]
    [clojure.java.io :as io]))

(defmethod aero/reader 'lower-case
  [_opts _tag value]
  (-> value str string/lower-case))

(defn- validate-key
  "Given the config, validator and the current key
   being validated, finds the validations function in
   the validation map and applies it to the config"
  [conf validator current-key]
  {:pre [(map? conf)
         (map? validator)
         (or (keyword? current-key)
             (string? current-key)
             (vector? current-key))]}
  (let [path-validator (get validator current-key)
        k-path (if (vector? current-key)
                 current-key
                 [current-key])
        conf-path-value (get-in conf k-path)
        validator-fn (cond
                       (fn? path-validator) path-validator
                       ;; map is a value validator so if not matching value, the value doesn't
                       ;; need to be validated
                       (map? path-validator) (or (get path-validator conf-path-value)
                                                 (constantly true)))]
    (and validator-fn (fn? validator-fn) (validator-fn conf-path-value conf))))

(defn read-config
  "Reads config from resources.
   Allowed options:
   - resolver: The resolver to use in aero (default: aero/resource-resolver)
   - profile: The profile to use is #profile is used
   - validate: A map of keys in the config to be validated.
               Keys can be vectors with the path to the config to be validated.
               Values can be a map with {<expected key value> <validation fn>} or simply <validation fn>
               All validators must return 'truthy' or a validation exception will be raised
   Example: for the config
   {:env 'prod' :host 'localhost'}
   Validation that host can't be localhost in prod can be added with:
   (read-config '<the conf file>' {:validate {:env {'prod' #(not (= 'localhost' (:host %2)))}}})"
  ([config-path]
   (read-config config-path {}))
  ([config-path
   {:keys [profile resolver validate]
    :or {resolver aero/resource-resolver}}]
  (when-let [conf (aero/read-config (io/resource config-path) {:resolver resolver :profile profile})]
    (when validate
      (mapv
        (fn validate-fn [k]
          (when-not (validate-key conf validate k)
            (throw
              (ex-info "invalid-config" {:validate-key k}))))
        (keys validate)))
    conf)))




