(ns grid.astar)

(defn heuristic [[x1 y1] [x2 y2]]
  ;; Manhattan distance for 4-way movement
  (+ (Math/abs (- x1 x2))
     (Math/abs (- y1 y2))))

(defn neighbors [[x y]]
  ;; 4-way movement
  (->> [[(dec x) y]
        [(inc x) y]
        [x (dec y)]
        [x (inc y)]]
       (filterv #(and (<= 0 (first %))
                      (<= 0 (second %))))))

(defn reconstruct-path [came-from current]
  (loop [cur current
         path [current]]
    (if-let [prev (came-from cur)]
      (recur prev (conj path prev))
      (reverse path))))

(defn a-star
  "Return path as a seq of coordinates from start to goal or nil if none.
   passable? is (pos -> boolean)."
  [start goal passable?]
  (loop [open-set #{start}
         came-from {}
         g-score {start 0}
         f-score {start (heuristic start goal)}]
    (if (empty? open-set)
      nil
      (let [current (apply min-key #(get f-score % ##Inf) open-set)]
        (if (= current goal)
          (reconstruct-path came-from current)
          (let [open-set (disj open-set current)
                g        (get g-score current 0)
                ;; process neighbors, threading updates through a reduce
                [open-set came-from g-score f-score]
                (reduce
                  (fn [[os cf gs fs] nbr]
                    (if-not (passable? nbr)
                      [os cf gs fs]
                      (let [tentative-g (inc g)]
                        (if (< tentative-g (get gs nbr ##Inf))
                          [(conj os nbr)
                           (assoc cf nbr current)
                           (assoc gs nbr tentative-g)
                           (assoc fs nbr (+ tentative-g (heuristic nbr goal)))]
                          [os cf gs fs]))))
                  [open-set came-from g-score f-score]
                  (neighbors current))]
            (recur open-set came-from g-score f-score)))))))

(comment
  (def walls #{[0 1]})
  (defn passable? [pos] (not (walls pos)))
  (a-star [0 0] [0 3] passable?)
  ;; => ([0 0] [0 1] [0 2] [0 3] [1 3] [2 3] [3 3])  (or another valid shortest path)
  )
