(ns george.javafx.inspector
  (:require [clojure.java.io :as cio]
            [clojure.string :as cs]
            [george.javafx.core :as fx] :reload
            [george.javafx.java :as j] :reload)
  (:import (javafx.scene.control TreeItem TreeView TextField TreeCell ContextMenu MenuItem)
           (javafx.scene.layout HBox)
           (javafx.scene Node)
           (clojure.lang PersistentArrayMap PersistentHashMap PersistentTreeMap Atom AMapEntry PersistentVector Seqable PersistentHashSet Namespace)
           (javafx.scene.paint Color)))





(defn expanded? [tree-cell]
  (if-let [item  (.getTreeItem tree-cell)]  ;; prevent NullPointerException
    (.isExpanded item)
    false))



(defn- namespace-content [ns]
  {
   :interns (ns-interns ns)
   :refers  (ns-refers ns)
   :aliases (ns-aliases ns)
   :imports (ns-imports ns)})


(defn- namespace? [x]
  (instance? Namespace x))


(defn- atom? [data]
  (instance? Atom data))


(defn- single? [x]
  (not (or
         (var? x)
         (and (atom? x) (coll? (deref x)))
         (namespace? x)
         (coll? x))))


(defn- coll-open-str [data]
  (cond
    (map? data) "{"
    (vector? data) "["
    (set? data) "#{"
    (or (list? data)(seq? data)) "("
    :default "?"))

(defn- coll-close-str [data]
  (cond
    (map? data) "}"
    (vector? data) "]"
    (set? data) "}"
    (or (list? data)(seq? data)) ")"
    :default (str "?" (type data))))


(defn- token-repr-str [item]
  (cond
    (char? item)
    (format "\\%s" item)

    (string? item)
    (format "\"%s\"" item)

    (nil? item)
    "nil"

    :default
    (str item)))


(defn- coll-repr-str [item is-expanded]
  (if is-expanded
      (coll-open-str item)
      (format "%s<%s>%s"
              (coll-open-str item)
              (count item)
              (coll-close-str item))))


(defn- item-repr-str
  "returns a string representing the item in a list"
  [item is-expanded]
  (cond
    (map-entry? item)
    (format "%s  %s" (item-repr-str (key item) false) (item-repr-str (val item) is-expanded))

    (coll? item)
    (coll-repr-str item is-expanded)

    (atom? item)
    (format "@ %s" (item-repr-str (deref item) is-expanded))

    :default
    (token-repr-str item)))



(definterface ItemRepr
  (data [])
  (listableString [is_expanded]))


(defrecord ItemDataContainer [data]
  ItemRepr
  (data [_] data)
  (listableString [_ is-expanded] (item-repr-str data is-expanded))
  Object
  (toString [_] (str data)))





(declare treeitem)


(defn- sub-item [data level]
  (cond
    (atom? data)
    (sub-item (deref data) level)  ;; recur

    (var? data)
    (sub-item (meta data) level)  ;; recur

    (and (namespace? data) (< level 2))  ;; prevent recursion on namespaces in meta
    (sub-item (namespace-content data) level)  ;; recur

    (coll? data)
    ;(doseq [d data] (item d (inc level))) ;; "append" children
    (for [d data] (treeitem d (inc level)))))



(defn treeitem

  ([data]
   (treeitem data 0))

  ([data level]
   ;; "create" item
   (println (str  (clojure.string/join (for [i (range level)] "--")) "  " (item-repr-str data true)))

   (let [
         item
         (proxy [TreeItem] [(ItemDataContainer. data)]
           (isLeaf []
             (or (single? data)
                 (and (atom? data) (single? (deref data)))
                 (and (map-entry? data) (single? (val data))))))

         children
         (.getChildren item)

         kids
         (if (map-entry? data)
             (sub-item (val data) level)
             (sub-item data level))]

     (doseq [k kids]
       (.add children k))

     item)))


(defn- expand-collapse-all [item is-expand]
  (.setExpanded item is-expand)
  (doseq [child (-> item .getChildren)]
    (expand-collapse-all child is-expand)))


(defn- contextmenu [item]
  (ContextMenu.
    (j/vargs
      (doto (MenuItem. "Expand all")
            (.setOnAction (fx/event-handler (j/thread (expand-collapse-all item true)))))
      (doto (MenuItem. "Collapse all")
            (.setOnAction (fx/event-handler (j/thread (expand-collapse-all item false))))))))



(defn cell-factory [items-to-highlight-set-atom]
  (reify javafx.util.Callback
    (call [this tree-view]
      (let [tree-cell
            (doto
              (proxy [TreeCell] []
                (updateItem [obj is-empty]
                  (proxy-super updateItem obj is-empty)
                  ;(println "  ## obj:" obj (type obj))
                  (if is-empty
                    ;; then
                    (doto this
                      (.setText nil)
                      (.setGraphic nil))
                    ;; else
                    (let [item  (.getTreeItem this)
                          txt   (.listableString obj  (expanded? this))]

                      (doto this
                        (.setText txt)
                        (.setContextMenu (contextmenu item)))

                      ;; look for tree-item in set
                      (if ((deref items-to-highlight-set-atom) item)
                        (.setBackground this (fx/color-background Color/YELLOW))
                        (.setBackground this nil))))))
              ;(->> .getTreeItem .getGraphic (.setGraphic this))
              (.setFont (fx/SourceCodePro "Regular" 12)))]

        tree-cell))))



(defn- search-for-item-in-tree [search-rgx item result-set-atom]
  (when (re-find search-rgx (-> item .getValue (.listableString true)))
    (swap! result-set-atom conj item)
    (.setExpanded item true))
  (doseq [child (-> item .getChildren)]
    (search-for-item-in-tree search-rgx child result-set-atom)))



(defn tree-inspector-pane
  [data-fn]
  (let [

        data (data-fn)
        items-to-highlight-set-atom (atom #{})

        root-item
        (doto (treeitem data)
              (.setExpanded true))

        tree-view
        (doto (TreeView. root-item)
          (.setCellFactory (cell-factory items-to-highlight-set-atom)))
          ;(fx/BorderPane_setMargin (fx/insets 20 0 0 0)))

        hide-field
        (doto (TextField. "")
          (.setPromptText "Search. Start typing ... (regex)")
          (-> .textProperty
              (.addListener
                (fx/changelistener [_ _ _ new-val]
                                   (j/thread
                                     (println "  ## new-val:" new-val)
                                     (reset! items-to-highlight-set-atom #{})
                                     (when-not (cs/blank? new-val)
                                       (search-for-item-in-tree (-> new-val cs/trim re-pattern) root-item items-to-highlight-set-atom))))))
          (.setPrefWidth 400))

        refresh-button-fn
        (fn []
          (println "refresh does nothing!"))
;          (.setRoot tree-view (tree-item (namespaces-map (.getText hide-field)))))

        refresh-button
        (fx/button "Refresh" :onaction #(refresh-button-fn))


        top-bar
        (fx/hbox
          (fx/label "Find?:") ;"Hide NS:")
          hide-field
          (fx/region :hgrow :always)
          refresh-button
          :spacing 5
          :insets [0 0 5 0])

        pane
        (fx/borderpane :top top-bar :center tree-view :insets 20)]

    pane))





(defn tree-inspector
  [data-fn]
  (let [stage
        (fx/now
          (fx/stage
            :scene (fx/scene (tree-inspector-pane data-fn))
            :size [800 600]
            :title "tree-inspector"))]
    stage))



;;;; DEV

(def a :A)
(def testdata {"array" [1 2 4]
               "boolean" true
               "null" nil
               :number 123
               :sub-map {
                         :a "b"
                         "c" \d
                         :string "Hello World"
                         :another-vector [10 1/2 -2.45]}
               :vector [1 2 3]
               :set #{:1 2 "3"}
               :list (list "a" "b")
               :seq (seq '(1 2 3))
               "atom" (atom "A value")
               "another-atom" (atom [3 4])
               :var (var a)
               :function (var tree-inspector)})

(println "WARNING. Running george.javafx.inspector/tree-inspector")
(tree-inspector (constantly testdata))
;(tree-inspector #(read-string (slurp "testdata/cms-xml.clj")))
;(tree-inspector (constantly (the-ns 'george.javafx.inspector)))
;(treeitem testdata)
;(treeitem (all-ns))
;(treeitem (the-ns 'george.javafx.inspector))
