/*
 *  Copyright (c) 2017 Terje Dahl. All rights reserved.
 * The use and distribution terms for this software are covered by the Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php) which can be found in the file epl-v10.html at the root of this distribution.
 *  By using this software in any fashion, you are agreeing to be bound by the terms of this license.
 *  You must not remove this notice, or any other, from this software.
 */

package jewelsea;

/* https://gist.github.com/jewelsea/5683558 */

import javafx.animation.Transition;
import javafx.beans.property.DoubleProperty;
import javafx.beans.value.*;
import javafx.collections.*;
import javafx.scene.Node;
import javafx.scene.transform.Translate;
import javafx.util.Duration;

import java.util.*;

/**
 * Animates an object when its position is changed. For instance, when
 * additional items are added to a Region, and the layout has changed, then the
 * layout animator makes the transition by sliding each item into its final
 * place.
 */
public class LayoutAnimator implements ChangeListener<Number>, ListChangeListener<Node> {

  private Map<Node, MoveXTransition> nodeXTransitions = new HashMap<Node, MoveXTransition>();
  private Map<Node, MoveYTransition> nodeYTransitions = new HashMap<Node, MoveYTransition>();

  /**
   * Animates all the children of a Region.
   * <code>
   *   VBox myVbox = new VBox();
   *   LayoutAnimator animator = new LayoutAnimator();
   *   animator.observe(myVbox.getChildren());
   * </code>
   *
   * @param nodes
   */
  public void observe(ObservableList<Node> nodes) {
    for (Node node : nodes) {
      this.observe(node);
    }
    nodes.addListener(this);
  }

  public void unobserve(ObservableList<Node> nodes) {
    nodes.removeListener(this);
  }

  public void observe(Node n) {
    n.layoutXProperty().addListener(this);
    n.layoutYProperty().addListener(this);
  }

  public void unobserve(Node n) {
    n.layoutXProperty().removeListener(this);
    n.layoutYProperty().removeListener(this);
  }

  @Override
  public void changed(ObservableValue<? extends Number> ov, Number oldValue, Number newValue) {
    final double delta = newValue.doubleValue() - oldValue.doubleValue();
    final DoubleProperty doubleProperty = (DoubleProperty) ov;
    final Node node = (Node) doubleProperty.getBean();

    String s = doubleProperty.getName();
    if (s.equals("layoutX")) {
      MoveXTransition tx = nodeXTransitions.get(node);
      if (tx == null) {
        tx = new MoveXTransition(node);
        nodeXTransitions.put(node, tx);
      }
      tx.setFromX(tx.getTranslateX() - delta);

      tx.playFromStart();

    } else {
      MoveYTransition ty = nodeYTransitions.get(node);
      if (ty == null) {
        ty = new MoveYTransition(node);
        nodeYTransitions.put(node, ty);
      }
      ty.setFromY(ty.getTranslateY() - delta);

      ty.playFromStart();
    }
  }

  private abstract class MoveTransition extends Transition {
    private final Duration MOVEMENT_ANIMATION_DURATION = new Duration(200);

    protected final Translate translate;

    public MoveTransition(final Node node) {
      setCycleDuration(MOVEMENT_ANIMATION_DURATION);
      translate = new Translate();

      node.getTransforms().add(translate);
    }

    public double getTranslateX() {
      return translate.getX();
    }

    public double getTranslateY() {
      return translate.getY();
    }
  }

  private class MoveXTransition extends MoveTransition {
    private double fromX;

    public MoveXTransition(final Node node) {
      super(node);
    }

    @Override protected void interpolate(double frac) {
      translate.setX(fromX * (1 - frac));
    }

    public void setFromX(double fromX) {
      translate.setX(fromX);
      this.fromX = fromX;
    }
  }

  private class MoveYTransition extends MoveTransition {
    private double fromY;

    public MoveYTransition(final Node node) {
      super(node);
    }

    @Override protected void interpolate(double frac) {
      translate.setY(fromY * (1 - frac));
    }

    public void setFromY(double fromY) {
      translate.setY(fromY);
      this.fromY = fromY;
    }
  }

  @Override
  public void onChanged(Change change) {
    while (change.next()) {
      if (change.wasAdded()) {
        for (Node node : (List<Node>) change.getAddedSubList()) this.observe(node);
      } else if (change.wasRemoved()) {
        for (Node node : (List<Node>) change.getRemoved()) this.unobserve(node);
      }
    }
  }

  // todo unobserving nodes should cleanup any intermediate transitions they may have and ensure they are removed from transition cache to prevent memory leaks.
}
