(ns george.appengine.lein
  (:require
    [clojure.pprint :refer [pprint]]
    [clojure.set :refer [difference]]
    [clojure.java.io :as cio]
    [leiningen.core.main :refer [info warn debug abort *cwd*] :as lein]
    ;[environ.core :refer [env]]
    [leiningen.core.classpath :as lclasspath]
    [leiningen.core.main :as lmain]
    [leiningen.repl :as lrepl]
    [cemerick.pomegranate :as pom]
    [cemerick.pomegranate.aether :as aether]
    [robert.hooke]
    [robert.hooke :as hooke]
    [leiningen.deps :as ldeps]
    [leiningen.core.project :as lproject])
    ;[leiningen.localrepo :refer [localrepo]])
    ;[lein-extend-cp.plugin :refer [add-more-cps]])
  (:import (java.util.zip ZipFile)
           (java.io File)
           (com.google.common.io Files ByteStreams)))




(def AE_SDK_FILE_NAME ".appengine-sdk")


(defn- entries [zipfile]
  (enumeration-seq (.entries zipfile)))



(defn- unzip [zipped-file target-parent-dir overwrite?]
  (let [sdk-zip-archive (ZipFile. ^File zipped-file)
        zip-entries (entries sdk-zip-archive)
        file-entries (remove #(.isDirectory %) zip-entries)
        first-entry (first zip-entries)
        sdk-base-dir-suffix (.getName first-entry)
        _ (debug "  ## sdk-base-dir-suffix:" sdk-base-dir-suffix)]

    (when (or
            (not (.exists (cio/file target-parent-dir sdk-base-dir-suffix)))
            overwrite?)
      (info "unzipping ...")
      (doseq [zip-entry file-entries]
        (let [zip-entry-destination (cio/file target-parent-dir (.getName zip-entry))]
          (when-not (.exists zip-entry-destination)
            (Files/createParentDirs zip-entry-destination)
            (Files/write (ByteStreams/toByteArray (.getInputStream sdk-zip-archive zip-entry))
                         zip-entry-destination))))

      (when-not (every? #(.startsWith (.getName %) sdk-base-dir-suffix) file-entries)
        target-parent-dir))

    (cio/file target-parent-dir sdk-base-dir-suffix)))



(defn- ^File resolve-sdk-zip
  "Downloads the GAE SDK zip-file if necessary.
 Returns the path to the downlaoded zip-file."
  [sdk-version]
  (let [coordinate ['com.google.appengine/appengine-java-sdk sdk-version :extension "zip"]

        _ (info "resolving" coordinate "...")
        artifact (first (aether/resolve-artifacts :coordinates [coordinate]))
        _ (debug "  ## artifact:" artifact)

        sdk-zip-file (:file (meta artifact))
        _ (debug "  ## sdk-zip-file:" sdk-zip-file)]

    (.getCanonicalFile sdk-zip-file)))


(defn ^File resolve-tools-api-jar [sdk-dir]
  (let [
        appengine-tools-api-jar (cio/file sdk-dir "lib" "appengine-tools-api.jar")
        _ (debug "  ## appengine-tools-api-jar:" appengine-tools-api-jar)
        _ (debug "  ## appengine-tools-api-jar exists:" (.exists appengine-tools-api-jar))]
    (.getCanonicalFile appengine-tools-api-jar)))



(defn- resolve-sdk
  "Resolves GAE SDK zip-file, downloading it  to repo if neccassary, unzips it.
Returns a map containing the path to the unzip-dir and the version downloaded:
{:sdk-path ... :sdk-version ...}"
  [project]
  (debug "sdk called ...")
  (let [sdk-version (-> project :appengine :sdk-version)
        _ (debug "  ## sdk-version:" sdk-version)
        sdk-zip-file (resolve-sdk-zip sdk-version)
        _ (debug "  ## sdk-zip-file:" sdk-zip-file)
        sdk-repo-dir (.getParentFile sdk-zip-file)
        _ (debug "  ## sdk-repo-dir:" sdk-repo-dir)
        sdk-dir (unzip sdk-zip-file sdk-repo-dir false)
        _ (info "SDK unzipped into:" (str sdk-dir))]

       {:sdk-path (str sdk-dir) :sdk-version sdk-version}))
            ;(doseq [r (lein-cp/get-classpath project)] (debug " - " r))))
            ;com.google.appengine.tools.development.DevAppServerMain))



(defn- resolve-sdk-and-jars
  "'resolves' GAE SDK and necessary jar(s), and writes them to .appengine in project-dir, returning SDK-dir"
  [project]
  (let [sdk-info (resolve-sdk project)
        ;tools-api-jar (resolve-tools-api-jar sdk-info)

        ;info {:sdk-dir sdk-dir :tools-api-jar tools-api-jar}
        ;info-stringed  (into {} (for [[keep v] info] [keep (str v)]))]
        sdk-info-stringed (into {} (for [[k v] sdk-info] [k (str v)]))]
    (clojure.pprint/pprint sdk-info-stringed (clojure.java.io/writer AE_SDK_FILE_NAME))
    (debug (format "  ## %s\n%s" AE_SDK_FILE_NAME (slurp AE_SDK_FILE_NAME)))
    sdk-info))



;(defn- classpath-hook [task & args]
;  (debug "  ## classpath-hook called")
;  (let [jr (:tools-api-jar (resolve-sdk-and-jars (first args)))
;        new-paths (concat (apply task args) [jr])]
;    new-paths))


(defn read-lein-project []
  (lproject/read))



(defn- verify-ae-sdk-data []
  (when-not (.exists (cio/file AE_SDK_FILE_NAME))
    (throw (Exception. ^String (format "Could not locate file %s (in %s).  Have you remembered to do `lein appengine sdk`?"  AE_SDK_FILE_NAME *cwd*))))
  true)



(defn- verify-ae-params []
  (let [min-text "Project must at minimum contain ':appengine {:sdk-version \"some-version\"}'"
        settings (:appengine (read-lein-project))]
    (when-not settings
      (throw (Exception. (str "Missing ':appengine' param in project.clj. " min-text))))
    (when-not (:sdk-version settings)
      (throw (Exception. (str "Missing ':sdk-version' param in ':appengine' param in project.clj. " min-text))))
    true))


(defn read-appengine-sdk []
  (when (verify-ae-sdk-data)
    (read-string (slurp AE_SDK_FILE_NAME))))


;; TODO: this doesn't get used, I don't think.
(defn read-appengine-settings []
  (when (verify-ae-params)
    (:appengine (read-lein-project))))



(defn sdk-path []
  (:sdk-path (read-appengine-sdk)))


(defn sdk-jars-map []
  (let [sdk-dir (cio/file (sdk-path))]
   {:appengine-tools-api
    (.getAbsoluteFile (cio/file sdk-dir  "lib" "appengine-tools-api.jar"))
    :appengine-api-stubs
    (.getAbsoluteFile (cio/file sdk-dir  "lib" "impl" "appengine-api-stubs.jar"))
    :appengine-local-runtime
    (.getAbsoluteFile (cio/file sdk-dir  "lib" "impl" "appengine-local-runtime.jar"))}))


(defn- sdk-jars-paths
  "returns a seq of path-strings"
  []
  (map #(str (val %)) (sdk-jars-map)))


(defn- classpath-hook
  "adds skd-jars to classpath"
  [task & args]
  (println "  PLN  classpath-hook called!")
  (concat (apply task args) (sdk-jars-paths)))


(defn dir?
  [d]
  (and (instance? File d)
       (.isDirectory d)))


(defn file?
  [f]
  (and (instance? File f)
       (.isFile f)))

(defn pwd []
  (.getParent (.getAbsoluteFile (File. "."))))


(defn assert-dir
  [^String dir]
  (if (dir? (cio/file dir))
    dir
    (abort (format "ERROR: '%s' is not a directory, current directory is: %s" dir (pwd)))))


(defn assert-file
  [^String file]
  (if (file? (cio/file file))
    file
    (abort (format "ERROR: '%s' is not a file, current directory is: %s" file (pwd)))))


(defn- install-sdk-jar [jar-path artifact-id version]
  (debug "  ## installining SDK jar:  " artifact-id)
  ;; [repo-path pom-filename filename artifact-id version]
  (aether/install ;:local-repo (assert-dir repo-path)
                  :coordinates [(symbol artifact-id) version]
                  :jar-file (cio/file (assert-file jar-path))))
                  ;:pom-file (assert-file pom-filename)))


(defn install-sdk-jars [{:keys [_ sdk-version] :as sdk-info}]
  (doseq [[k jar-file] (seq (sdk-jars-map))]
    (install-sdk-jar
      (str jar-file)
      (format "com.google.appengine/%s" (name k))
      sdk-version)))



;;;;  "public" ;;;;

(defn ammend-classpath
  "adds sdk-jar-paths to existing classpath, if missing."
   []
  (let [system-classpaths (pom/get-classpath)
        sdk-jar-classpaths (sdk-jars-paths)
        missing-paths (difference (set sdk-jar-classpaths) (set system-classpaths))]
    (doseq [p missing-paths]
      (pom/add-classpath p))))


;(defn middleware
;  "Adds necessary GAE SDK files
;Called by leiningen via george-appengine-lein.plugin/middleware."
;  [project]
;  (debug "  ## my 'middleware' got called"))
;
;  ;(pom/add-classpath (resolve-tools-api-jar-file project))
;  ;(-> project (update-in [:resource-paths] concat [(resolve-tools-api-jar-file project)])))

(defn- sdk
  [project]
  (let [sdk-info (resolve-sdk-and-jars project)]
    (install-sdk-jars sdk-info)
    sdk-info))


(defn- deps-hook [task & args]
  (debug "  ## deps-hook called")
  (sdk (first args))
  (apply task args))






(defn sdk-task
  "A leiningen subtask that will pull and unzip the GAE SDK from Maven.
The SDK version must be listed in project.clj as:
   :appengine {:sdk-version \"some-version\"}  ;; i.e. :appengine {:sdk-version \"1.9.42\"}.
(Is perhaps also called automatically by middleware.)"
  [project]
  (info "GAE SDK dir:" (str (:sdk-path (sdk project)))))









