(ns george.appengine.devserver
  (:require
    [clojure.java.io :as cio]
    [clojure.reflect :refer [reflect]]
    [clojure.pprint :refer [pprint]]
    [cemerick.pomegranate :as pom]
    [clojure.string :as cs]
    [george.appengine.lein :as gal])
  (:import (java.util.jar JarFile)
           (clojure.lang Reflector)
           (java.io File)))




(defn list-jar-content [jar-file-or-path]
  (let [jar (JarFile. jar-file-or-path)]
    (println "listing content of jar:" jar-file-or-path)
    (doseq [entry (enumeration-seq (.entries jar))]
      ;(when (re-find #"com/google/appengine/tools" (str entry))
      (println "  " entry))))



;;;; "public" ;;;;


(defn load-class [^String class-name]
  (let [class-name class-name
        class-loader (.getContextClassLoader (Thread/currentThread))
        loaded-class (.loadClass class-loader class-name)]
    loaded-class))


(defn proxy-from-class [^Class cls proxy-init-map]
  (let [pc (get-proxy-class cls)
        p (construct-proxy pc)]
    (init-proxy p proxy-init-map)))


(defn proxy-from-jar [^String jar-path ^String canonical-class proxy-init-map]
  (pom/add-classpath jar-path)
  (let [cls (load-class canonical-class)]
    (proxy-from-class cls proxy-init-map)))


(defonce srv (atom nil))


;; To find/kill server process on port on Windows (in DOS Terminal):
;; 1.  `netstat -aon`
;; 2. locate PID for process on port
;; 3. `taskkill /F /PID <PID>


(def DEFAULT_SETTINGS {:address "localhost" :port 3000 :app-dir-path "target/war"})


(defn devserver [& {:as args-map}]
  (let [
        ;; We don't use KickStarter, as it is intended for starting a server as a seperate system process.
        ;;   https://code.google.com/p/googleappengine/source/browse/trunk/java/src/main/com/google/appengine/tools/KickStart.java

        ;; Also, we don't use DevAppServerMain, as it too starts a separate system process.
        ;; In stead we in replicate the minimum required based on the classes StartAction.apply()
        ;;   https://code.google.com/p/googleappengine/source/browse/trunk/java/src/main/com/google/appengine/tools/development/DevAppServerMain.java#224

        ;; We need a few jars in our path - from the GAE SDK.

        sdk-dir (:sdk-dir (gal/read-appengine-sdk))

        ;; and lets get the settings from the lein-project and merge with default and passed-in
        project-settings (gal/read-appengine-settings)
        settings (conj DEFAULT_SETTINGS project-settings args-map)

        ;; for com.google.appengine.tools.info.AppengineSdk
        appengine-tools-api-jar-path (.getAbsolutePath (cio/file sdk-dir  "lib" "appengine-tools-api.jar"))
        _ (pom/add-classpath appengine-tools-api-jar-path)

        ;; don' need theses ... for now
        ;agent-jar-path (str (cio/file sdk-dir "lib" "agent" "appengine-agent.jar"))
        ;override-jar-path (str (cio/file sdk-dir "lib" "override" "appengine-dev-jdk-overrides.jar"))
        ;impl-appengine-api-jar-path (.getAbsolutePath (cio/file sdk-dir  "lib" "impl" "appengine-api.jar"))
        ;impl-appengine-api-labs-jar-path (.getAbsolutePath (cio/file sdk-dir  "lib" "impl" "appengine-api-labs.jar"))

        ;; for com.google.appengine.api.backends.dev.LocalServerController
        impl-appengine-api-stubs-jar-path (.getAbsolutePath (cio/file sdk-dir  "lib" "impl" "appengine-api-stubs.jar"))
        _ (pom/add-classpath impl-appengine-api-stubs-jar-path)
        ;_ (load-class "com.google.appengine.api.backends.dev.LocalServerController")
        ;; for com.google.appengine.tools.development.DevAppServerMainImpl
        ;;   https://code.google.com/p/googleappengine/source/browse/trunk/java/src/main/com/google/appengine/tools/development/DevAppServerImpl.java
        ;; and com.google.appengine.tools.development.DevAppServerFactory
        ;;   https://code.google.com/p/googleappengine/source/browse/trunk/java/src/main/com/google/appengine/tools/development/DevAppServerFactory.java?r=530
        impl-appengine-local-runtime-jar-path (.getAbsolutePath (cio/file sdk-dir  "lib" "impl" "appengine-local-runtime.jar"))
        _ (pom/add-classpath impl-appengine-local-runtime-jar-path)

        ;; We use the DevAppServerFactory to create a DevAppServer instance.
        ;; We might as well proxy it, as we have a nice neat function for that.
        ;; (And we might want to extend something in future.)
        ;;   https://code.google.com/p/googleappengine/source/browse/trunk/java/src/main/com/google/appengine/tools/development/DevAppServerFactory.java?r=530
        cls (load-class "com.google.appengine.tools.development.DevAppServerFactory")
        init-map {}
        factory (proxy-from-class cls init-map)
        server (.createDevAppServer factory (cio/file (:app-dir-path settings))  nil (:address settings) (:port settings) true)]
    ;; We also need to set a couple service properties minimum for it to work.
    ;; The args need to match between the factory and the services!
    (.setServiceProperties server {"address" (:address settings) "port" (str (:port settings))})
    server))



(defn server
  "inits a new server, stopping any previous server if present."
  [& params]
  (when @srv
    (.shutdown @srv))
  (reset! srv (apply devserver params)))


(defn restart []
  (if @srv
    (.restart @srv)
    (println "Ops!  No server set up.  Call `(server & params)` first.")))


(defn start []
  (if @srv
    (.start @srv)
    (println "Ops!  No server set up.  Call `(server & params)` first.")))


(defn stop []
  (if @srv
    (.shutdown @srv)
    (println "Ops!  No server to stop.  Call `(server & params)` first.")))



;(defn serve
;  "The only command you ever need:
;   It will set up a server if not already.
;  It will start server, if not started.
;  It will restart server, if started.
;  It will set up new server if nil, or current server stopped."
;  [& params])
