(ns rill.event-store.psql
  (:require [clojure.core.async :refer [thread <!! >!! chan close!]]
            [clojure.java.jdbc :as sql]
            [clojure.tools.logging :as log]
            [rill.event-store :refer [EventStore]]
            [rill.event-stream :refer [all-events-stream-id]]
            [rill.message :as message]
            [rill.uuid :refer [uuid]]
            [taoensso.nippy :as nippy])
  (:import [java.util Date]
           [java.sql Timestamp SQLException BatchUpdateException]))

(defn record->metadata
  [r]
  (let [m (transient {message/number (:stream_order r)
                      message/cursor (:insert_order r)})
        typed (if-let [type-str (:event_type r)]
                (assoc! m message/type (keyword type-str))
                m)
        stamped (if-let [timestamp ^Timestamp (:created_at r)]
                  (assoc! typed message/timestamp (Date. (.getTime timestamp)))
                  typed)
        partitioned (if-let [partition-id (:partition_id r)]
                      (assoc! stamped message/partition-id partition-id)
                      stamped)
        id-d (if-let [id (:event_id r)]
               (assoc! partitioned message/id (uuid id))
               partitioned)
        streamed (if-let [stream-id (:stream_id r)]
                   (assoc! id-d message/stream-id stream-id)
                   id-d)]
    (persistent! streamed)))


(defn record->message
  [r]
  (merge (-> r :payload nippy/thaw)
         (record->metadata r)))

(defn wrap-auto-retry
  [f pause-seconds]
  (fn [& args]
    (loop [args args]
      (let [r (try
                (apply f args)
                (catch Exception e
                  (log/error e (str "Exception, will retry in " pause-seconds " seconds."))
                  ::retry))]
        (if (= r ::retry)
          (do (Thread/sleep (* 1000 pause-seconds))
              (recur args))
          r)))))

(def retrying-query (wrap-auto-retry sql/query 30))

(defn select-stream-query [stream-order stream-id page-size]
  ["SELECT * FROM rill_events WHERE stream_id = ? AND stream_order > ? ORDER BY stream_order ASC LIMIT ?"
   (str stream-id) stream-order page-size])

(defn select-all-query [cursor page-size]
  ["SELECT * FROM rill_events WHERE insert_order > ? ORDER BY insert_order ASC LIMIT ?"
   cursor page-size])

(defn select-partition-query [cursor partition-id page-size]
  ["SELECT * FROM rill_events WHERE partition_id = ? AND insert_order > ? ORDER BY insert_order ASC LIMIT ?"
   (str partition-id) cursor page-size])

(defn messages
  [cursor page-size selector ordering-fn]
  (let [p (mapv record->message (selector cursor page-size))]
    (if (< (count p) page-size)
      (lazy-seq p)
      (concat p (lazy-seq (messages (ordering-fn (peek p))
                                    page-size
                                    selector
                                    ordering-fn))))))

(defn unique-violation?
  "true when the exception was caused by a unique constraint violation"
  [sql-exception]
  (= (.getSQLState sql-exception) "23505"))


(defn catch-up-events
  [spec]
  (log/debug "Catch-up-events")
  (let [page-size 100 ;; arbitrary
        in (chan 3)
        out (chan 3)]
    ;; db thread
    (thread
      (loop [cursor -1]
        (let [rows (retrying-query spec
                                   (select-all-query cursor page-size)
                                   {:result-set-fn vec})
              highest-insert-order (:insert_order (peek rows))]
          (>!! in rows)
          (log/debug "[db] Passed along upto :insert_order" highest-insert-order " cursor: " cursor)
          (if (< (count rows) page-size)
            (log/debug "[db] Got a batch with less than page-size events, done with reading db [old-cursor highest-insert-order]" [cursor highest-insert-order])
            (recur highest-insert-order))))
      (log/debug "[db] Finished catching up from db")
      (close! in))

    ;; deserializer thread
    (thread
      (log/debug "[deserializer] Starting deserializer")
      (loop []
        (when-let [rows (<!! in)]
          (log/debug "[deserializer] Deserializing from " (:insert_order (first rows)) " upto " (:insert_order (peek rows)))
          (let [transformed (mapv record->message rows)]
            (>!! out transformed))
          (recur)))
      (close! out)
      (log/debug "[deserializer] Stopped deserializer"))

    ;; lazy-seq
    (let [out-seq (fn out-seq []
                    (if-let [events (<!! out)]
                      (concat events (lazy-seq (out-seq)))
                      (do (log/debug "[lazy-seq] Ending lazy-seq from catch-up events")
                          nil)))]
      (log/debug "[lazy-seq] Delivering to lazy-seq")
      (out-seq))))

(defn type->str
  [k]
  (subs (str k) 1))

(defn strip-metadata
  [e]
  (dissoc e message/type message/id message/number message/timestamp message/stream-id message/cursor message/partition-id))

(defrecord PsqlEventStore [spec page-size]
  EventStore
  (retrieve-events-since [this stream-id cursor _wait-for-seconds]
    (if (and (= stream-id all-events-stream-id)
             (= cursor -1))
      ;; catching up case
      (catch-up-events spec)
      ;; listening after catching up and aggregate case
      (let [cursor (if (number? cursor)
                     cursor
                     (or (message/cursor cursor)
                         (throw (ex-info (str "Not a valid cursor: " cursor) {:cursor cursor}))))]
        (if (= stream-id all-events-stream-id)
          (messages cursor
                    page-size
                    (fn [cursor page-size] (sql/query spec (select-all-query cursor page-size)))
                    message/cursor)
          (messages cursor
                    page-size
                    (fn [cursor page-size] (sql/query spec (select-stream-query cursor stream-id page-size)))
                    message/number)))))

  (retrieve-partitions [this]
    (set (map :partition_id (retrying-query spec ["SELECT DISTINCT partition_id from rill_events"]))))

  (retrieve-events-from-partition [this partition-id cursor]
    (messages cursor
              page-size
              (fn [cursor page-size] (sql/query spec (select-partition-query cursor partition-id page-size)))
              message/cursor))

  (append-events [this stream-id partition-id from-version events]
    (assert partition-id "partition-id can not be nil")
    (try (if (= from-version -2) ;; generate our own stream_order
           (do (sql/with-db-transaction [conn spec]
                 (sql/db-do-prepared conn false (into ["INSERT INTO rill_events (event_id, stream_id, stream_order, partition_id, created_at, event_type, payload) VALUES (?, ?, (SELECT(COALESCE(MAX(stream_order),-1)+1) FROM rill_events WHERE stream_id=?), ?, ?, ?, ?)"]
                                                      (map (fn [e]
                                                             [(str (message/id e))
                                                              (str stream-id)
                                                              (str stream-id)
                                                              (str partition-id)
                                                              (Timestamp. (.getTime (message/timestamp e)))
                                                              (type->str (message/type e))
                                                              (nippy/freeze (strip-metadata e))])
                                                           events))
                                     {:multi? true})
                 true))
           (try (sql/with-db-transaction [conn spec]
                  (sql/db-do-prepared conn (into ["INSERT INTO rill_events (event_id, stream_id, stream_order, partition_id, created_at, event_type, payload) VALUES (?, ?, ?, ?, ?, ?, ?)"]
                                                 (map-indexed (fn [i e]
                                                                [(str (message/id e))
                                                                 (str stream-id)
                                                                 (+ 1 i from-version)
                                                                 (str partition-id)
                                                                 (Timestamp. (.getTime (message/timestamp e)))
                                                                 (type->str (message/type e))
                                                                 (nippy/freeze (strip-metadata e))])
                                                              events))
                                      {:multi? true})
                  true)
                (catch BatchUpdateException e ;; conflict - there is already an event with the given stream_order
                  (when-not (unique-violation? e)
                    (throw e))
                  false)))
         (catch SQLException e
           (if-let [next-exception (.getNextException e)]
             (throw next-exception)
             (throw e))))))

(defn psql-event-store [spec & [{:keys [page-size] :or {page-size 20}}]]
  {:pre [(integer? page-size)]}
  (let [es (->PsqlEventStore spec page-size)]
    (assoc es :store es)))
