(ns rill.event-store.memory
  "An in-memory event store for testing purposes"
  (:require [clojure.tools.logging :as log]
            [rill.event-store :as store]
            [rill.event-stream :refer [all-events-stream-id empty-stream-version any-stream-version empty-stream]]
            [rill.message :as message]
            [slingshot.slingshot :refer [try+ throw+]]))

(deftype MemoryStore [state]
  store/EventStore
  (retrieve-events-since [this stream-id cursor wait-for-seconds]
    (subvec (get-in @state [:by-stream-id stream-id] empty-stream) (inc cursor)))

  (retrieve-partitions [this]
    (get @state :partitions #{}))

  (retrieve-events-from-partition [this partition-id since]
    (subvec (get-in @state [:by-partition-id partition-id] empty-stream) (inc since)))

  (append-events [this stream-id partition-id from-version events]
    (assert partition-id)
    (try+ (swap! state (fn [old-state]
                         (let [current-stream (get-in old-state [:by-stream-id stream-id] empty-stream)
                               cursor (get old-state :cursor -1)
                               current-version (if (empty? current-stream)
                                                 empty-stream-version
                                                 (message/number (peek current-stream)))
                               start-number (if (= from-version any-stream-version)
                                              current-version
                                              from-version)

                               events (mapv #(assoc %1
                                                    message/number (+ start-number %2)
                                                    message/cursor (+ cursor %2)
                                                    message/stream-id stream-id
                                                    message/partition-id partition-id)
                                            events
                                            (iterate inc 1))]
                           (if (= (dec (count current-stream)) start-number)
                             (-> old-state
                                 (update-in [:by-stream-id stream-id] (fnil into []) events)
                                 (update-in [:by-partition-id partition-id] (fnil into []) events)
                                 (update :partitions (fnil conj #{}) partition-id)
                                 (update :cursor (fnil + -1) (count events))
                                 (update-in [:by-stream-id all-events-stream-id] (fnil into []) events))
                             (throw+ ::out-of-date)))))
          true
          (catch #(= % ::out-of-date) err
            nil))))

(defn memory-store
  []
  (MemoryStore. (atom {})))
