;;; SPDX-FileCopyrightText: 2024 Jomco B.V.
;;; SPDX-FileContributor: Joost Diepenmaat <joost@jomco.nl>
;;; SPDX-FileContributor: Remco van 't Veer <remco@jomco.nl>
;;;
;;; SPDX-License-Identifier: MIT

(ns nl.jomco.resources)

(defprotocol Resource
  "Protocol defining a closeable resource.

  Resources are opened items that must be closed. Any
  java.lang.AutoCloseable (including java.io.Closeable) is a
  resource.

  Components that are started and should be stopped can extend
  Resource (via metadata or using extend-protocol) to implement the
  `close` method."
  :extend-via-metadata true
  (close [resource] "Close resource."))

;; resources
;;
;; - must be closed in reverse order of opening
;;
;; - must be closed even if exceptions occur (even when closing other
;;   resources) -- implies (try .. finally)
;;
;; - can be opened, used, closed explicitly (as in with-open)
;;
;; - need direct access to resource (makes wrapping objects
;;   inconvenient)
;;
;; - can compose -- encapsulate construction/start and closed -- TBD?
;;
;; - dependency injection conflicts with encapsulation -- should a
;;   provided dependency be closed when dependent resource is closed?
;;   -- probably not.
;;
;; - when encapsualated resources are started this may fail and they
;;   should be closed. But if this succeeds... looks like RAII / ref
;;   counting destructors.
;;
;; x (f...)
;; y (g ..)
;; z (f x y)
;;
;; - threads? What if I want to start a service in a REPL? I want to
;;   do this in a separate thread, and be able to stop the service
;;   from the repl. I can't easily use with-open in this case...
;;
;;  (def something (open! ...))
;;
;;   ...
;;
;;  (close something)
;;
;; Maybe we should distinguish resources (that can be local bindings
;; and be closed automatically) from services (that maybe should be
;; registered for REPL? -- make sure you can't forget to bind a
;; service to a global?
;;
;; What if we define a defresource that closes the resource when
;; rebound? this should also deal with cases when namespace was
;; deleted (see mount)
;;
;;
;;

(extend-protocol Resource
  java.lang.Object
  (close [_])
  java.lang.AutoCloseable
  (close [resource]
    (.close resource)))

(defmacro with-resources
  "bindings => [name init ...]

  Evaluates body in a try expression with names bound to the values of
  the inits, and a finally clause that calls Resource
  protocol's (close name) on each name in reverse order.

  This is equivalent to clojure's `with-open`, but allows objects to
  specify `close` behaviour via `Resource` protocol.

  Like `with-open`, when exceptions are thrown during closing, the
  last thrown exception is raised from `with-resources`.

  `with-resources` does not close bindings when the JVM is shut
  down. See `close-on-shutdown!`."
  [bindings & body]
  (cond
    (empty? bindings)
    `(do ~@body)

    (not (even? (count bindings)))
    (throw (IllegalArgumentException.
            "with-resources bindings must be an even number of forms"))

    (symbol? (bindings 0))
    `(let ~(subvec bindings 0 2)
       (try
         (with-resources ~(subvec bindings 2) ~@body)
         (finally
           (close ~(bindings 0)))))

    :else
    (throw (IllegalArgumentException.
            "with-resources only allows Symbols in bindings"))))

(defn closeable
  "Returns `x` with an `nl.jomco.resources/close` function added.

  Result is a resource. `x` must be a clojure object that supports
  metadata. When `x` is closed, `close-fn` will be called with `x` as
  its only argument. If `x` already provides a close function, it will
  be called after `close-fn`.

  If `close-fn` is not provided, `identity` will be used as
  `close-fn`.

      (with-resources [x (closeable {:my \"object\") cleanup-object)]
         ...)"
  ([x close-fn]
   (let [old-x x]
     (vary-meta x assoc `close (fn [new-x]
                                 (try (close-fn new-x)
                                      (finally
                                        (close old-x)))))))
  ([x]
   (closeable x identity)))

(defn close-on-shutdown!
  "Register a shutdown hook that will close the resource.

  When the JVM is shut down, the resource's close method will be
  called.

  Returns resource with an additional close function that will remove
  the shutdown hook. This means the following will work and clean up
  the hook if the JVM was not shut down during the `with-resources`
  call:

      (with-resources [r (close-on-shutdown! (open-some-resource))]
        (do-some-long-running-thing-with r))

  See also `closeable`, `defresource`."
  [resource]
  (let [thread  (Thread. (fn []
                           (close resource)))
        runtime (Runtime/getRuntime)]
    (.addShutdownHook runtime thread)
    (closeable resource
               (fn [_]
                 (.removeShutdownHook runtime thread)))))

(defmacro defresource
  "Creates and interns a global var that will hold a resource.

  `defresource` is intended for REPL use. Like `clojure.core/def`; var
  will be named by `symbol`. If `init-resource` is provided, it is
  evaluated and the root-binding of the var is set to the result. A
  `doc-string` may be provided.

  Registers a shutdown hook to close the resource, with
  `close-on-shutdown!`.

  If var already holds a resource, it will be closed before evaluating
  `init-resource` and assigning the new resource. This will also
  unregister the shutdown hook for the previous resource.

      (defresource my-db
         \"A database connection\"
         (init-db ...))"
  {:arglists '[[symbol doc-string? init-resource?]]}
  [& args]
  (when-not (<= 1 (count args) 3)
    (throw (IllegalArgumentException. "defresource takes 1 to 3 arguments")))
  `(do (some-> '~(first args)
               resolve
               deref
               close)
       ~(if (< 1 (count args))
          `(def ~@(butlast args)
             (close-on-shutdown! ~(last args)))
          `(def ~(first args)))))

(defmacro mk-system
  "Bind opened resources to symbols in order of appearance.

  If an exception is raised, ensures that opened resources are closed
  in reverse order and re-raises the exception.

  The result of the evaluation -- a system, which can be any clojure
  IObj, normally a clojure collection -- is wrapped in a `closeable`
  that will close all bound resources, so that when the returned
  object is closed, all the bound resources are closed in reverse
  order of binding.

  To provide access to the bound resources, it is recommeneded to
  return the resources in a system map or record:

      (mk-system [db (open-db ...) ;; initialize resources
                  h  (create-handler db) ;; pass dependencies
                  s  (run-server handler)]
        ;; return map of sub-resources to use as the system
        {:db db
          :handler h
          :server s})

  If no `body` is provided, a map is returned with key-value pairs
  generated from `bindings`:

      (mk-system [db (open-db ...) ;; initialize resources
                  h  (create-handler db) ;; pass dependencies
                  s  (run-server handler)])
      ;; => {:db .. :h ... :s ..}
"
  [bindings & body]
  (cond
    (empty? body)
    `(mk-system ~bindings
       ~(into {}
              (map (fn [sym]
                     [(keyword (name sym)) sym])
                   (take-nth 2 bindings))))

    (empty? bindings)
    `(do ~@body)

    (simple-symbol? (first bindings))
    `(let ~(subvec bindings 0 2)
       (try (mk-system ~(subvec bindings 2)
              (closeable (do ~@body)
                         (fn [_#]
                           (close ~(first bindings)))))
            (catch java.lang.Throwable t#
              (close ~(first bindings))
              (throw t#))))

    :else
    (throw (IllegalArgumentException. "bindings must be symbols."))))

(defn wait-until-interrupted
  "Block the current thread until it's interrupted.

  If the thread is interrupted, returns `true`. Clears the interrupted
  status of the thread."
  []
  (loop []
    (when-not (try (Thread/sleep 60000)
                   false
                   (catch InterruptedException _
                     true))
      (recur))))
