(ns nl.jomco.select-tree
  "Select subtrees of collections.

  A straightforward way to describe and create subselections of nested
  collections.

  # Design Considerations

  ## Clojury interface

  Selectors should not look out of place in idiomatic clojure code;
  selectors are plain data.

  ## General applicability

  - Any combination of standard clojure collections and leaf nodes can
    be selected
    - Vectors (x number of items from the head)
    - Maps
    - Sets
    - Lists (x number of items from the head)

  ## Selected structure should be a \"subtree\" of the original

  - Selected elements in a collection remain on the same key or index;
    paths to items in the collection remain valid in the selection.
  - Selectors are unambiguous; the meaning of a selector does not depend
    on the structure of the tree.
  - Selection of a collection will return the same kind of collection.
  - Selection does not add keys to collections or extend sequential
    collections.
  - Selection retains metadata.

  ## Selectors should be unambigious

  - Paths in a selection can only be specified once, to prevent
    ambiguity."
  (:require [clojure.set :as set]))

(defn select-tree
  "Return a selection from `coll` specified by `selector`.

  Seletors are defined recursively:

  A `nil` selector returns coll as is.

  A map selector recursively selects items in a map coll, with selectors
  as values.

  A set selector works as a map selector with `nil` values (select whole
  value for each key). Set selectors also work on sets.

  A sequential selector of size N recursively selects the first N, or
  less if the collection is smaller, items in the sequential collection,
  with selectors as values.

  An integer selector N selects the first N items in a sequential
  collection (like a sequential selector with N `nil` entries).

  Returns a selection of the same type as `coll`, with the same
  metadata."
  [coll selector]
  (cond
    (nil? selector)
    coll

    (nil? coll)
    nil

    (map? selector)
    (if (map? coll)
      (reduce (fn [m [k s]]
                (if-let [[_ v] (find coll k)]
                  (assoc m k (select-tree v s))
                  m))
              (empty coll)
              selector)
      (throw (ex-info "Map selector not valid for coll"
                      {:coll coll
                       :selector selector})))

    (set? selector)
    (cond
      (map? coll)
      (select-keys coll selector)

      (set? coll)
      (set/intersection coll selector)

      :else
      (throw (ex-info "Set selector not valid for coll"
                      {:coll coll
                       :selector selector})))

    (sequential? selector)
    (cond
      (seq? coll)
      (map select-tree coll selector)

      (sequential? coll)
      (into (empty coll)
            (map select-tree coll selector))

      :else
      (throw (ex-info "Sequential selector not valid for coll"
                      {:coll coll
                       :selector selector})))

    (pos-int? selector)
    (cond
      (seq? coll)
      (take selector coll)

      (sequential? coll)
      (into (empty coll)
            (take selector coll))

      :else
      (throw (ex-info "Numeric selector not valid for coll"
                      {:coll coll
                       :selector selector})))

    :else
    (throw (ex-info "Not a valid selector type" {:selector selector
                                                 :coll coll}))))
