(ns nl.jomco.ring-trace-context
  (:require [clojure.string :as string])
  (:import org.apache.commons.codec.binary.Hex
           java.util.Random))

(defn parse-traceparent
  "Parse `traceparent` string and return a trace-context map.


  This will fully parse version \"00\" traceparent values.

  trace-context map contains :version, :trace-id, :parent-id
  and :trace-flags (a set of keywords).

  The only currently supported trace-flag is :sampled.

  Returns `nil` if `traceparent` is not a valid traceparent value.

  See also https://www.w3.org/TR/trace-context"
  [traceparent]
  (when (string? traceparent)
    (when-let [[_ version] (re-find #"^([0-9a-f]{2})-" traceparent)]
      (when (and (not= "ff" version) ;; valid version
                 (>= (count traceparent) 55))  ;; valid header size
        (when-let [[_ trace-id parent-id flags]
                   (re-find #"^[0-9a-f]{2}-([0-9a-f]{32})-([0-9a-f]{16})-([0-9a-f]{2})" traceparent)]
          (when (and (not= "00000000000000000000000000000000" trace-id) ;; invalid trace-id
                     (not= "0000000000000000" parent-id) ;; invalid parent-id
                     )
            (let [flag-bits (Long/parseLong flags 16)
                  sampled?  (= 0x01 (bit-and flag-bits 0x01))]
              {:version     version
               :trace-id    trace-id
               :parent-id   parent-id
               :trace-flags (if sampled? #{:sampled} #{})})))))))

(defn generate-traceparent
  "Returns a traceparent header value from the given `trace-context`."
  {:arglists '([trace-context])}
  [{:keys [version trace-id parent-id] {:keys [sampled]} :trace-flags :or {version "00"}}]
  (str version "-" trace-id "-" parent-id "-" (if sampled "01" "00")))

(defn parse-tracestate
  [s]
  (when s
    (let [list-members (-> (if (coll? s) (string/join "," s) s)
                           (string/trim)
                           (string/split #"[\t ]*,[\t ]*"))]
      (mapv #(string/split % #"=" 2) list-members))))

(defn- random-hex
  [num-bytes]
  (let [buff (byte-array num-bytes)]
    (.nextBytes (Random.) buff)
    (Hex/encodeHexString buff)))

(defn random-context
  "Create new random trace-context.

  :trace-id and :parent-id are random. :version is always 00
  and :trace-flags are empty."
  []
  {:version     "00"
   :trace-id    (random-hex 16)
   :parent-id   (random-hex 8)
   :trace-flags #{}})

(def ^:dynamic
  *trace-context*
  "The trace-context in the current scope.

  This will be set by `wrap-trace-context` automatically. Use
  `with-context` otherwise."
  nil)

(defn new-context
  "Returns a new context off the given `base-trace-context`.

  The returned trace-context has the same properties as the given
  `base-trace-context` but with a random :parent-id.

  If `base-trace-context` is `nil`, returns a random context.

  If no argument is passed, uses `*trace-context*` as the base."
  ([base-trace-context]
   (if base-trace-context
     (assoc base-trace-context :parent-id (random-hex 8))
     (random-context)))
  ([]
   (new-context *trace-context*)))

(defn new-span
  "Same as `new-context`."
  ([trace-context]
   (new-context trace-context))
  ([]
   (new-context)))

(defn get-context
  "Get the trace-context from the given ring `request-or-response`.

  Will return `nil` if no traceparent header is present."
  [request-or-response]
  (-> request-or-response
      (get-in [:headers "traceparent"])
      parse-traceparent))

(defn set-context
  "Set the traceparent header for `trace-context` on `request-or-response`."
  [request-or-response trace-context]
  (assoc-in request-or-response
            [:headers "traceparent"]
            (generate-traceparent trace-context)))

(defmacro with-context
  "Bind *trace-context* to `trace-context` and evaluate `body`."
  [trace-context & body]
  `(binding [*trace-context* ~trace-context]
     ~@body))

(defn wrap-trace-context
  "Ring middleware for Trace Context.

  Implements the W3C Trace Context protocol.  When a request has a
  `traceparent` header, this gets parsed into a `trace-context` which
  is set on the request, before passing it to `handler`.

  The parsed `trace-context` is set on the request and response as the
  `:trace-context` key.

  If no `traceparent` is set, or `traceparent` is not valid, a new
  random `trace-context` is created.

  Sets the response `traceparent` header to the trace-context used.

  During call to `handler`, `*trace-context*` is also bound to
  `trace-context`.

  See  https://www.w3.org/TR/trace-context/"
  [handler]
  (fn [request]
    (let [trace-context (or (get-context request) (random-context))]
      (with-context trace-context
        (-> (assoc request :trace-context trace-context)
            (handler)
            (assoc :trace-context trace-context)
            (set-context trace-context))))))
