(ns nl.jomco.proof-specs
  (:require [clojure.spec.alpha :as s]
            [clojure.test.check.generators :as gen]
            [clojure.tools.cli :as cli]))

(s/def ::regexp
  (s/with-gen
    #(instance? java.util.regex.Pattern %)
    #(gen/fmap re-pattern (s/gen string?))))

(s/def ::include (s/coll-of ::regexp))
(s/def ::exclude (s/coll-of ::regexp))
(s/def ::require (s/coll-of symbol?))
(s/def ::num-vals int?)
(s/def ::limit-ms int?)
(s/def ::verbose boolean?)

(s/def ::selector
  (s/keys :opt-un [::include ::exclude]))

(s/def ::error
  (s/with-gen
    #(instance? Exception %)
    #(gen/fmap (fn [s] (ex-info s {:msg s})) (s/gen string?))))

(s/def ::problems
  (s/map-of ::spec ::error))

(s/def ::spec qualified-keyword?)
(s/def ::specs
  (s/coll-of ::spec))

(s/fdef matcher
  :args (s/cat :selector ::selector)
  :ret fn?)

(defn- matcher
  "Return a matcher predicate given a selector.

  Selector is a map of :include and :exclude collections of regexps,
  both optional.

  Matcher returns true if given a qualified keyword of which _the
  fully qualified name_ matches the selector. A match is made if any
  of :include regexps matches the namespace and none of :exclude
  regexps matches the namespace.

  If :include is omitted, matches anything not matched by
  :exclude. If both are omitted, matches all qualified keywords."
  [{:keys [include exclude]}]
  (fn [n]
    (when (qualified-keyword? n)
      (let [ns (namespace n)]
        (and (or (empty? include)
                 (some #(re-matches % ns) include))
             (not-any? #(re-matches % ns) exclude))))))

(s/fdef select-specs
  :args (s/cat :selector ::selector)
  :ret ::specs)

(defn- select-specs
  "Select a set of specs matching `selector`. See also `matcher`."
  [selector]
  (set (filter (matcher selector)
               (map first (s/registry)))))

(defmacro ^:private timed
  [& body]
  `(let [s# (System/nanoTime)
         ret# (do ~@body)]
     {:msecs (/ (double (- (System/nanoTime) s#))
                1000000.0)
      :val ret#}))

(defn- exception-data
  [e]
  (cond-> {::message (.getMessage e)
           ::type (class e)}

    (ex-data e)
    (merge (ex-data e))

    (.getCause e)
    (assoc ::cause (exception-data (.getCause e)))))

(defn- generator-problems
  ([spec n]
   (try (doall (s/exercise spec n))
        nil
        ;; catching Throwable instead of Exception, since we want to
        ;; catch AssertionErrors amongst other things.
        (catch Throwable t
          [spec (exception-data t)])))
  ([spec n limit-ms]
   (let [{:keys [msecs val]} (timed (generator-problems spec n))]
     (cond
       (> limit-ms msecs) val

       val (update val 1 assoc
                   ::limit-ms limit-ms
                   ::generating-msecs msecs)

       :else [spec {::message "Time limit exceeded"
                    ::limit-ms limit-ms
                    ::generating-msecs msecs}]))))

(s/def ::proof-opts (s/keys* :opt-un [::include ::exclude
                                      ::num-vals ::require
                                      ::verbose ::limit-ms]))

(s/fdef proof-specs
  :args ::proof-opts
  :ret  (s/keys :req-un [::specs]
                :opt-un [::problems]))

(defn proof-specs
  "Check data specs in the selected namespaces.

  Exercises each keyword spec and reports on problems generating data.

  Namespace selection happens by optional kw args `:include` and
  `:exclude`, taking collections of regular expressions that match the
  names of the namespaces. If both args are empty, selects all loaded
  specs.

  If argument `:require` is provided, requires those namespaces prior
  to testing the specs.

  If `:num-vals` is provided, attempts to generate that many values
  from each spec (default is 10).

  If `:verbose` is provided and true, log to STDERR while proofing.

  If more than `:limit-ms` have elapsed while generating values for a
  spec, this is reported as a problem. Default is 100 milliseconds.

  Returns a map of

    - :specs - a collection of all specs checked
    - :problems - map of problems, if any problems are found

  Problems is a map of the problematic spec's key to the exception
  thrown when generating data for the spec."
  [{:keys [num-vals require verbose limit-ms] :as selector :or {num-vals 10 limit-ms 100}}]
  (doseq [n require]
    (when verbose
      (.println *err* (str "Requiring " n)))
    (clojure.core/require n))
  (let [ss (select-specs selector)]
    (if-let [problems (seq (keep #(do
                                    (when verbose
                                      (.println *err* (str "Proofing " %)))
                                    (generator-problems % num-vals limit-ms)) ss))]
      {:problems (into {} problems)
       :specs (set ss)}
      {:specs (set ss)})))

(def ^:private cli-opts
  [["-i" "--include REGEXP" "Include specs with qualified name matching REGEXP"
    :default #{} :update-fn conj
    :parse-fn #(java.util.regex.Pattern/compile %)
    :multi true]
   ["-x" "--exclude REGEXP" "Exclude specs with qualified matching REGEXP"
    :parse-fn #(java.util.regex.Pattern/compile %)
    :default #{} :update-fn conj
    :multi true]
   ["-v" "--verbose" "Be more verbose while running"]
   ["-r" "--require NAMESPACE" "Require NAMESPACE before proofing"
    :parse-fn symbol
    :default [] :update-fn conj
    :multi true]
   ["-n" "--num-vals NUMBER" "Generate NUMBER values per spec"
    :parse-fn #(Integer/parseInt %) :default 10]
   ["-m" "--limit-ms NUMBER" "Treat taking more than NUMBER milliseconds per spec as a problem" :parse-fn #(Integer/parseInt %) :default 100]
   ["-h" "--help"]])

(defn -main
  "Run [[proof-specs]] from CLI arguments.

  Run with --help for a report on all available options.

  Example

  ```
  lein run -m nl.jomco.proof-specs --num-vals 10 \\
       --include '.*nl.jomco.*' \\
       --require nl.jomco.proof-specs
  ```

  --include, --exclude and --require can be specified multiple times.

  Will print a report to STDOUT and exit. Status code will be 1 if
  problems are found, 0 otherwise."
  [& args]
  (let [{:keys [errors options summary arguments]} (cli/parse-opts args cli-opts)]
    (when errors
      (.println *err* errors)
      (System/exit 1))

    (when (seq arguments)
      (.println *err* (str "Takes no additional arguments (" arguments " provided)"))
      (System/exit 1))

    (when (:help options)
      (println summary)
      (System/exit 0))

    (let [{:keys [specs problems]} (proof-specs options)]
      (if problems
        (do
          (println "Problems generating data for" (count problems) "out of" (count specs) "specs:")
          (run! println (sort (keys problems)))
          (println)
          (prn problems))
        (println "No problems generating data for" (count specs) "specs."))
      (System/exit (if problems 1 0)))))
