(ns nl.jomco.proof-specs
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as string]
            [clojure.test.check.generators :as gen]))

(s/def ::regexp
  (s/with-gen
    #(instance? java.util.regex.Pattern %)
    #(gen/fmap re-pattern (s/gen string?))))

(s/def ::include-regexps (s/coll-of ::regexp))
(s/def ::exclude-regexps (s/coll-of ::regexp))
(s/def ::require-namespaces (s/coll-of symbol?))
(s/def ::num-vals int?)

(s/def ::selector
  (s/keys :opt-un [::include-regexps ::exclude-regexps]))

(s/def ::error
  (s/with-gen
    #(instance? Exception %)
    #(gen/fmap (fn [s] (ex-info s {:msg s})) (s/gen string?))))

(s/def ::problems
  (s/map-of ::spec ::error))

(s/def ::spec qualified-keyword?)
(s/def ::specs
  (s/coll-of ::spec))


(s/fdef matcher
  :args (s/cat :selector ::selector)
  :ret fn?)

(defn- matcher
  "Return a matcher predicate given a selector.

  Selector is a map of :include-regexps and :exclude-regexps, both
  optional.

  Matcher returns true if given a qualified keyword of which _the
  namespace_ matches the selector. A match is made if any of
  include-regexps matches the namespace and none of exclude-regexps
  matches the namespace.

  If include-regexps is omitted, matches anything not matched by
  exclude-regexps. If both are omitted, matches all qualified
  keywords."
  [{:keys [include-regexps exclude-regexps]}]
  (fn [n]
    (when (qualified-keyword? n)
      (let [ns (namespace n)]
        (and (or (empty? include-regexps)
                 (some #(re-matches % ns) include-regexps))
             (not-any? #(re-matches % ns) exclude-regexps))))))


(s/fdef select-specs
  :args (s/cat :selector ::selector)
  :ret ::specs)

(defn- select-specs
  "Select a set of specs matching `selector`. See also `matcher`."
  [selector]
  (set (filter (matcher selector)
               (map first (s/registry)))))

(defn- generator-problems
  [spec n]
  (try (doall (s/exercise spec n))
       nil
       (catch Exception e
         [spec e])))

(s/def ::proof-opts (s/keys* :opt-un [::include-regexps ::exclude-regexps
                                      ::num-vals ::require-namespaces]))

(s/fdef proof-specs
  :args ::proof-opts
  :ret  (s/keys :req-un [::specs]
                :opt-un [::problems]))

(defn proof-specs
  "Check data specs in the selected namespaces.

  Exercises each keyword spec and reports on problems generating data.

  Namespace selection happens by optional kw args `:include-regexps`
  and `:exclude-regexps`, taking collections of regular expressions
  that match the names of the namespaces. If both args are empty,
  selects all loaded specs.

  If argument `:require-namespaces` is provided, requires those
  namsepaces prior to testing the specs.

  If `:num-vals` is provided, attempts to generate that many values
  from each spec (default is 10).

  Returns a map of

    - :specs - a collection of all specs checked
    - :problems - map of problems, if any problems are found

  Problems is a map of the problematic spec's key to the exception
  thrown when generating data for the spec."
  [& {:keys [num-vals require-namespaces] :as selector :or {num-vals 10}}]
  (doseq [n require-namespaces]
    (require n))
  (let [ss (select-specs selector)]
    (if-let [problems (seq (keep #(generator-problems % num-vals) ss))]
      {:problems (into {} problems)
       :specs (set ss)}
      {:specs (set ss)})))

(defn -main
  "Run [[proof-specs]] from CLI arguments.

  Example

  ```
  lein run -m nl.jomco.proof-spec-gen --num-vals 10 \\
       --include-regexps '.*nl.jomco.*' \\
       --require-namespaces nl.jomco.proof-spec-gen
  ```

  Leading dashes `--` are optional for each argument. include-regexps,
  exclude-regexps and require-namespaces take a comma-separated
  string.

  Will print a report to STDOUT and exit. Status code will be 1 if
  problems are found, 0 otherwise."
  [& args]
  (when-not (even? (count args))
    (throw (ex-info "Takes an even number of args (key value pairs)" {:args args})))

  (let [opts (mapcat (fn [[k v]]
                       (let [k (-> k
                                   (string/replace #"^[\-:]+" "")
                                   keyword)
                             v (case k
                                 (:include-regexps :exclude-regexps)
                                 (mapv #(java.util.regex.Pattern/compile %)
                                       (string/split v #" *, *"))

                                 :num-vals
                                 (Integer/parseInt v)

                                 :require-namespaces
                                 (mapv symbol (string/split v #" *, *")))]
                         [k v]))
                     (partition 2 args))
        {:keys [specs problems]} (apply proof-specs opts)]
    (if problems
      (do
        (println "Problems generating data for" (count problems) "out of" (count specs) "specs:")
        (run! println (sort (keys problems)))
        (println)
        (prn problems))
      (println "No problems generating data for" (count specs) "specs."))
    (System/exit (if problems 1 0))))
