(ns nl.jomco.openapi.v3.validator.json-coerce
  {:no-doc true})

;; In JSON Schema (according to the JSON Schema Test Suite), numeric
;; semantics are slightly different from Java / Clojure:
;;
;; - There is only one Number type for decimals and integers, so 10.0
;;   is equal to 10.
;; - No complex numbers
;;
;; Implications:
;;
;; - `[5.0]` is equal to `[5]`
;; - `[5, 5.0]` does not have unique entries
;; - `{"a": 5}` is equal to `{"a": 5.0}`
;; - `5.0` validates according to `{"type": "integer"}`
;;
;; Further more, some of the JSON Schema Test Suite expects exact
;; results for operations (multipleOf) on decimal numbers (BigDecimal
;; semantics).
;;
;; Affected JSON Schemas keywords:
;;
;; - `const`
;; - `unique`
;; - `enum`
;; - `type`
;; - `multipleOf`
;;
;; Java / Clojure JSON parsers tend to parse `5.0` as a double, and
;; `5` as a long, and you can't rely on a `{"type": "integer"}` schema
;; validation to restrict input -- `5.0` is explictly valid for
;; `{"type": "integer"}`.
;;
;; Note: `clojure.data.json` has a `:bigdec` option, but that only
;; applies to numbers with decimal parts. Non-decimal numbers are
;; always parsed as `long` or `BigInteger`.
;;
;; You need a separate coercion mechanism if
;; you expect only longs or BigDecimals.
;;
;; Assuming we want to implement JSON Schema validation with the
;; semantics of JSON Schema Test Suite and we want to work with
;; current JSON parsers we have to process the instances before
;; applying some validations.
;;
;; The solution implemented here is to define `json-coerce` to coerce
;; numbers (as scalars and in collections) to BigDecimals, and then
;; compare using the coerced values.
;;
;; The tradeoff we chose is that we have to coerce as we validate,
;; which may be costly if we need to coerce large collections multiple
;; times, but:
;;
;; - we don't need to rely on pre-processed data or new parser
;;
;; - we only need to compare full datastructures for "unique"
;;   and "const" validations. We assume the instances to be validated
;;   using these keywords are typically scalars or small structures.
;;
;; If we want to use the previous, stricter, "JVM style" validation
;; that considers doubles unequal to long, we could provide that as an
;; option (not implemented).

(defn json-coerce
  [x]
  (cond
    (number? x)
    (bigdec x)

    (sequential? x)
    (into (empty x)
          (map json-coerce)
          x)

    (map? x)
    (reduce-kv (fn [m k v]
                 (assoc m k (json-coerce v)))
               (empty x)
               x)
    :else
    x))
