(ns nl.jomco.openapi.v3.util
  "Internal utility functions for implementing OpenAPI and JSON-Schema
  handling."
  {:no-doc true})

(defmacro delayedfn
  "Define a function that will be evaluated at first use.

  The `body` expression is wrapped in a delay and should evaluate to a
  function.

  When the delayed function is called for the first time, `body` is
  evaluated and the result is cached, and then called with the
  provided arguments.

  Subsequent calls will re-use the cached body function.

  This differs from `memoize` since it caches the function
  implementation, instead of its return value."
  [& body]
  `(let [impl# (delay ~@body)]
     (fn [& args#]
       (apply @impl# args#))))

(defmacro cached-at!
  "Return item at `path` in `cache-atom`, creating it if necessary.

  If there is no item at `path`, evaluates `body` to create it."
  [cache-atom path & body]
  `(let [path# ~path
         cache# ~cache-atom]
     (swap! cache# update-in path#
            (fn [cached#]
              (if (some? cached#)
                cached#
                (do ~@body))))
     (get-in @cache# path#)))
