(ns nl.jomco.openapi.v3.validator
  (:require
   [clojure.string :as string]
   [nl.jomco.openapi.v3.coercer :as coercer]
   [nl.jomco.openapi.v3.json-pointer :as pointer]
   [nl.jomco.openapi.v3.media-matcher :refer [media-matcher]]
   [nl.jomco.openapi.v3.path-matcher :refer [paths-matcher]]
   [nl.jomco.openapi.v3.util :refer [cached-at!]]
   [nl.jomco.openapi.v3.validator.json-schema-validator :as schema]))

;; WIP

(defn- normalize-uri
  [uri-prefix uri]
  (let [uri (string/replace uri #"^[^:]*://[^/]*" "")
        uri (if (and uri-prefix (string/starts-with? uri uri-prefix))
              (string/replace-first uri uri-prefix "")
              uri)
        uri (if (= \/ (first uri))
              uri
              (str "/" uri))]
    uri))

(defn- parameter-path
  [in n]
  (case in
    "query"
    [:query-params (name n)]

    "path"
    [:path-params (name n)]

    "header"
    [:headers (string/lower-case (name n))]

    "cookie"
    [:cookies (name n)]))

(defn- parameter-validator
  [{:keys [specification] :as context} canonical-schema-path]
  (when-let [[canonical-schema-path {:strs [name in required schema]}] (pointer/find specification canonical-schema-path)]
    (let [schema-validator
          (if schema
            (schema/schema-validator context
                                     (conj canonical-schema-path "schema"))
            (constantly nil))

          p
          (parameter-path in name)

          coerce
          (coercer/parameter-coercer context canonical-schema-path)]
      (fn [request path schema-path]
        (let [instance (get-in request p ::not-found)
              path     (into path p)]
          (if (= ::not-found instance)
            (when required
              [{:canonical-schema-path canonical-schema-path
                :issue                 "parameter-error"
                :path                  path
                :schema                {"name"     name
                                        "required" required}
                :schema-key            "required"
                :schema-path           schema-path}])
            (let [[instance issues] (coerce instance path (conj schema-path "schema"))]
              (if (seq issues)
                issues
                (schema-validator instance path (conj schema-path "schema"))))))))))

(defn- media-type-validator*
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[canonical-schema-path {:strs [schema]}]
        (pointer/find specification canonical-schema-path true)]
    (if schema
      (let [validate-body (schema/schema-validator context (conj canonical-schema-path "schema"))]
        (fn [request-or-response path schema-path]
          ;; TODO: coerce / parse body
          (validate-body (:body request-or-response) (conj path :body) (conj schema-path "schema"))))
      (constantly nil) ;; any request body is ok
      )))

(defn- media-type-validator
  [{:keys [cache] :as context} canonical-schema-path]
  (cached-at! cache
              [::media-type-validators canonical-schema-path]
              (media-type-validator* context canonical-schema-path)))

(defn- request-body-validator
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[canonical-schema-path {:strs [required content]}]
        (pointer/find specification canonical-schema-path true)
        ranges      (keys content)
        match-media (media-matcher ranges)]
    (fn [request path schema-path]
      (let [content-type (get-in request [:headers "content-type"])]
        (if-let [range (match-media content-type)]
          ((media-type-validator context
                                 (into canonical-schema-path ["content" range]))
           request path (into schema-path ["content" range]))
          (when required
            [{:canonical-schema-path canonical-schema-path
              :hints                 {:ranges ranges}
              :instance              content-type
              :issue                 "content-type-error"
              :path                  (into path  [:headers "content-type"])
              :schema-path           schema-path}]))))))

(defn- request-operation-validator*
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[params-path parameters]
        (pointer/find specification (conj canonical-schema-path "parameters"))

        request-body-path
        (pointer/canonical-path specification (conj canonical-schema-path "requestBody"))]
    (schema/checks
     (into (if request-body-path
             [(-> (request-body-validator context request-body-path)
                  (schema/with-schema-path-section ["requestBody"]))]
             [])
           (->> (range (count parameters))
                (map #(-> (parameter-validator context (conj params-path %))
                          (schema/with-schema-path-section ["parameters" %]))))))))

(defn- request-operation-validator
  [{:keys [cache] :as context} canonical-schema-path]
  (->> (request-operation-validator* context canonical-schema-path)
       (cached-at! cache [::operation-validators canonical-schema-path])))

(defn- uri-error
  [instance path paths]
  {:canonical-schema-path ["paths"]
   :instance              instance
   :issue                 "uri-error"
   :path                  (into path [:request :uri])
   :hints                 {:paths paths}
   :schema-path           ["paths"]})

(defn- method-error
  [method template path paths]
  {:canonical-schema-path ["paths" template]
   :hints                 {:methods (keys (get paths template))}
   :instance              method
   :issue                 "method-error"
   :path                  (into path [:request :method])
   :schema-path           ["paths" template]})

(defn request-validator
  [{:keys [uri-prefix specification] :as context}]
  (let [paths   (keys (get specification "paths"))
        matcher (paths-matcher paths)]
    ;; TODO: normalize requests (method, body, params)
    (fn [{:keys [uri method] :as request} path]
      (let [uri (normalize-uri uri-prefix uri)]
        (if-let [{:keys [template parameters]} (matcher uri)]
          (let [schema-path ["paths" template (name method)]]
            (if-let [canonical-schema-path (pointer/canonical-path specification schema-path)]
              ((request-operation-validator context canonical-schema-path)
               (assoc request :path-params parameters)
               path
               schema-path)
              [(method-error method template path paths)]))
          [(uri-error uri path paths)])))))

;; for requests, parameter-validator is used to check headers
(defn- response-header-validator
  [{:keys [specification] :as context} canonical-schema-path]
  (let [name (last canonical-schema-path)]
    (when-let [[canonical-schema-path {:strs [required schema]}]
               (pointer/find specification canonical-schema-path)]
      (let [schema-validator
            (if schema
              (schema/schema-validator context
                                       (conj canonical-schema-path "schema"))
              (constantly nil))

            coerce
            (coercer/header-coercer context canonical-schema-path name)]
        (fn [response path schema-path]
          (let [instance (get-in response [:headers name] ::not-found)
                path     (into path [:headers name])]
            (if (= ::not-found instance)
              (when required
                [{:canonical-schema-path canonical-schema-path
                  :issue                 "parameter-error"
                  :path                  path
                  :schema                {"name"     name
                                          "required" required}
                  :schema-path           schema-path}])
              (let [[instance issues] (coerce instance path schema-path)]
                (if (seq issues)
                  issues
                  (schema-validator instance path schema-path))))))))))

(defn- response-object-validator*
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[headers-path headers]
        (pointer/find specification
                      (conj canonical-schema-path "headers"))

        headers-validator
        (when headers
          (->> (keys headers)
               (map #(-> (response-header-validator context (conj headers-path %))
                         (schema/with-schema-path-section ["headers" %])))
               (schema/checks)))

        [content-path content]
        (pointer/find specification
                      (conj canonical-schema-path "content"))

        match-media (media-matcher (keys content))]
    (fn [{{:strs [content-type]} :headers :as response} path schema-path]
      (if-let [range (match-media content-type)]
        (cond->
            ((media-type-validator context (conj content-path range))
             response path (into schema-path ["content" range]))
          headers-validator
          (schema/combine-issues (headers-validator response path schema-path)))
        [{:canonical-schema-path content-path
          :hints                 {:ranges (keys content)}
          :instance              content-type
          :issue                 "content-type-error"
          :path                  (into path [:headers "content-type"])
          :schema-path           (conj schema-path "content")}]))))

(defn- response-object-validator
  [{:keys [cache] :as context} canonical-schema-path]
  (cached-at! cache
              [::response-object-validators canonical-schema-path]
              (response-object-validator* context canonical-schema-path)))

(defn- status-to-range
  [status]
  (str (first (str status))
       "XX"))

(defn- status-matcher
  [ranges]
  (let [ranges (set ranges)]
    (->> (range 100 600)
         (keep (fn [status]
                 (when-let [range (or (ranges (str status))
                                      (ranges (status-to-range status))
                                      (ranges "default"))]
                   [status range])))
         (into {}))))

(defn- response-operation-validator*
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[responses-path responses]
        (pointer/find specification
                      (conj canonical-schema-path "responses")
                      true)

        match-status
        (status-matcher (keys responses))]
    (fn [{:keys [status] :as response} path schema-path]
      (if-let [range (match-status status)]
        ((response-object-validator context (conj responses-path range)) response path (into schema-path ["responses" range]))
        [{:canonical-schema-path responses-path
          :hints                 {:ranges (keys responses)}
          :instance              status
          :issue                 "status-error"
          :path                  (conj path :status)
          :schema-path           schema-path}]))))

(defn- response-operation-validator
  [{:keys [cache] :as context} canonical-schema-path]
  (cached-at! cache
              [::response-operation-validators canonical-schema-path]
              (response-operation-validator* context canonical-schema-path)))

(defn response-validator
  [{:keys [uri-prefix specification] :as context}]
  (let [paths   (keys (get specification "paths"))
        matcher (paths-matcher paths)]
    ;; TODO: normalize requests (method, body, params)
    (fn [{:keys [request response]} path]
      (let [{:keys [uri method]} request
            uri                  (normalize-uri uri-prefix uri)]
        (if-let [{:keys [template]} (matcher uri)]
          (let [op-path ["paths" template (name method)]]
            (if (pointer/canonical-path specification op-path)
              ((response-operation-validator context op-path) response (conj path :response) op-path)
              [(method-error method template path paths)]))
          [(uri-error uri path paths)])))))

(defn interaction-validator
  [{:keys [uri-prefix specification] :as context}]
  (let [response-val (response-validator context)
        request-val  (request-validator context)
        paths-map    (get specification "paths")
        paths        (keys paths-map)
        matcher      (paths-matcher paths)]
    (fn [{:keys [request] :as interaction} path]
      (let [{:keys [uri method]} request
            uri                  (normalize-uri uri-prefix uri)]
        (if-let [{:keys [template]} (matcher uri)]
          (if (pointer/canonical-path specification
                                      ["paths" template (name method)])
            (schema/combine-issues
             (request-val request (conj path :request))
             (response-val interaction path))
            [(method-error method template path (paths-map path))])
          [(uri-error uri path paths)])))))

(defn validator-context
  "Create a new validator context from the given `specification`.

   `opts` is a map of options (or nil).

   Available options:

    - :format-predicates, a map of format values to format predicates,
      for validation. When provided, validates instances according to the
      format predicates.  Default is `nil`.  Use
      `nl.jomco.openapi.v3.validator.json-schema-validator/format-predicates`
      to enable validation of format specs.

    - :numeric-coercion, a function to coerce collections and scalars
      for comparing validations (enum, const, multiple-of and
      unique). Default is
      `nl.jomco.openapi.v4.validator.json-coerce/json-coerce`, which
      treats all numbers as BigDecimals (so instance `10.0` is valid
      according to `{\"const\": 10}`. Use `identity` to use Clojure
      semantics (where instance `[1, 1.0]` is valid according to
      `{\"uniqueItems\": true}`)."
  [specification opts]
  (schema/validator-context specification opts))
