(ns nl.jomco.openapi.v3.media-matcher
  (:require
   [clojure.string :as string])
  (:import
   (java.util.regex Pattern)))

(defn- parse-media-type
  [media-type]
  (string/split media-type #"\s*[/;]\s*"))

(defn- media-range-specifity
  [range]
  (let  [[type subtype & opts] (parse-media-type range)]
    (+ (cond
         (= "*" type)
         0
         (= "*" subtype)
         1
         :else
         2)
       (if (seq opts)
         0.5
         0))))

(defn- media-range-pattern
  [range]
  (let [[type subtype & _] (parse-media-type range)]
    (cond
      (= "*" type)
      #".*"

      (= "*" subtype)
      (re-pattern (str (Pattern/quote type)
                       "\\s*/*+"))

      :else
      (re-pattern (str (Pattern/quote type)
                       "\\s*/\\s*"
                       (Pattern/quote subtype)
                       ".*")))))

(defn- media-range-matcher
  [range]
  (let [pattern (media-range-pattern range)]
    (fn [media-type]
      (when (re-matches pattern media-type)
        range))))

(defn media-matcher
  "Find the matching media-type range.

  Given a collection media-type ranges, return a matcher that given a
  media-type, returns the most specific matching media-type-range, or
  nil."
  [ranges]
  (let [matchers (->> ranges
                      ;; most specific first
                      (sort-by #(- (media-range-specifity %)))
                      (mapv media-range-matcher))]
    (fn [media-type]
      (when media-type
        (some #(% media-type) matchers)))))
