(ns nl.jomco.openapi.v3.util
  "Internal utility functions for implementing OpenAPI and JSON-Schema
  handling.")

(defn kebab
  "Quick and dirty camelcase to kebab case converter."
  [s]
  (reduce (fn [out c]
            (if (Character/isUpperCase c)
              (str out "-" (Character/toLowerCase c))
              (str out c)))
          s))

(defn full-name
  "Returns the full name of k.

  - if k is a string, returns k
  - if k is a qualified key, returns \"<namespace>/<name>\"
  - if k is non-namespaced key, returns the name"
  [k]
  {:pre [(or (string? k) (keyword? k))]}
  (cond
    (string? k)
    k

    (qualified-keyword? k)
    (str (namespace k) "/" (name k))

    (keyword? k)
    (name k)))

(defmacro delayedfn
  "Define a function that will be evaluated at first use.

  The `body` expression is wrapped in a delay and should evaluate to a
  function.

  When the delayed function is called for the first time, `body` is
  evaluated and the result is cached, and then called with the
  provided arguments.

  Subsequent calls will re-use the cached body function.

  This differs from `memoize` since it caches the function
  implementation, instead of its return value."
  [& body]
  `(let [impl# (delay ~@body)]
     (fn [& args#]
       (apply @impl# args#))))

(defmacro cached-at!
  "Return item at `path` in `cache-atom`, creating it if necessary.

  If there is no item at `path`, evaluates `body` to create it."
  [cache-atom path & body]
  `(let [path# ~path
         cache# ~cache-atom]
     (swap! cache# update-in path#
            (fn [cached#]
              (if (some? cached#)
                cached#
                (do ~@body))))
     (get-in @cache# path#)))
