(ns nl.jomco.openapi.v3.path-matcher
  (:require
   [clojure.string :as string]
   [nl.jomco.openapi.v3.util :refer [full-name]]))

;; given a set of openapi templated paths match a url.
;;
;; Prefer non-templated paths over templated paths, by ordering paths
;; by their non-templated parts (longest parts first).


(defn- tokenize-template
  [template]
  (when template
    (let [[prefix rst] (string/split (full-name template) #"\{" 2)]
      (if rst
        (let [[param rst] (string/split rst #"\}" 2)
              tokens (list prefix (keyword param))]
          (if (not= "" rst)
            (concat tokens (tokenize-template rst))
            tokens))
        (list prefix)))))

(defn- token-specifity
  [s]
  (cond
    (string? s)
    (count s)

    (keyword? s)
    -1

    (nil? s)
    0))

(defn- compare-token
  [x y]
  (- (compare (token-specifity x) (token-specifity y))))

(defn- compare-tokens
  [x y]
  (if (= x y) ;; also catch x = nil, y = nil
    0
    (let [r (compare-token (first x) (first y))]
      (if (zero? r)
        (recur (next x) (next y))
        r))))

(defn compare-templates
  [x y]
  (compare-tokens (tokenize-template x) (tokenize-template y)))

(defn- tokens-pattern
  [tokens]
  (->> tokens
       (map (fn [token]
              (if (string? token)
                (java.util.regex.Pattern/quote token) ;; match literal string
                "([^/?#]*)")))
       (apply str)
       re-pattern))

(defn path-matcher
  "Return a matcher function that given a uri path, if the uri matches
  `template`, returns a map of :template and :parameters for the
  uri. Matcher returns `nil` if uri does not match."
  [template]
  (let [tokens          (tokenize-template template)
        path-parameters (filter keyword? tokens)
        regex (tokens-pattern tokens)]
    (fn [uri]
      (when-let [[_ & vals] (re-matches regex uri)]
        {:template   template
         :parameters (into {}
                           (map vector (map name path-parameters) vals))}))))

(defn paths-matcher
  "Return a matcher given a collection of `templates`.

  When matchers is given a uri, for the most specific template
  matching uri, return a map of :template and :parameters. Returns
  `nil` when no template matches."
  [templates]
  (let [matchers (->> templates
                      (sort compare-templates) ;; sort by most specific
                      (map path-matcher))]
    (fn [uri]
      (some #(% uri) matchers))))
