(ns nl.jomco.openapi.v3.validator.json-schema-validator
  (:require
   [nl.jomco.json-pointer :as pointer :refer [as-key]]
   [nl.jomco.openapi.v3.util :refer [cached-at! delayedfn kebab]]))

(defn unique?
  [items]
  (loop [seen  (transient #{})
         items items]
    (if (seq items)
      (if (contains? seen (first items))
        false
        (recur (conj! seen (first items)) (next items)))
      true)))

(defn validator-context
  [specification]
  {:specification specification
   :cache         (atom {})})

(defn combine-issues
  [i1 i2]
  (if (and i1 i2)
    (into i1 i2)
    (or i1 i2)))

(defn checks
  "Combine a collection of checks taking the same instance and path into
  a single check."
  [coll]
  (reduce (fn [res check]
            (if (and res check)
              (fn [instance instance-path]
                (combine-issues (res instance instance-path)
                                (check instance instance-path)))
              (or res check)))
          (constantly nil)
          coll))

(defmulti key-validator
  "Compile a check for the given JSON Schema key.

  Path must be a canonical path to an existing key"
  (fn [_context path]
    (last path))
  :default nil)

(defmethod key-validator nil
  [_context _schema-path]
  (fn [_instance _instance-path] nil))

(defn schema-validator
  [{:keys [specification cache] :as context} schema-path]
  {:pre [context (some? specification) schema-path]}
  (let [[schema-path schema]
        (pointer/find specification schema-path true)]
    (cond
      ;; boolean schema's always fail or always succeed
      (true? schema)
      (constantly nil)

      (false? schema)
      (fn [instance path]
        [{:issue       :schema-error
          :instance    instance
          :in          path
          :schema-path schema-path
          :schema      false}])

      (map? schema)
      ;; we need a delay since schemas can be defined
      ;; recursively (properties of a schema can use the
      ;; parent schema). This enables us to compile the
      ;; schema validator only once, the first time the
      ;; validator is invoked
      (->> (delayedfn
            (->> (keys schema)
                 (map #(key-validator context
                                      (conj schema-path %)))
                 checks))
           ;; we cache the validator in the context so we can reuse
           ;; the same schema validator when reached from multiple
           ;; `$ref`s
           (cached-at! cache [::schema-validators schema-path]))

      :else
      (throw (ex-info (str "Can't parse schema" (pr-str schema))
                      {:schema      schema
                       :schema-path schema-path})))))

(defn- mk-check
  [{:keys [specification] :as context} schema-path check-fn]
  {:pre [context specification schema-path (fn? check-fn)]}
  (let [k          (last schema-path)
        schema-val (get-in specification schema-path)
        issue-type (keyword (str (kebab (name k)) "-error"))
        c          (check-fn schema-val)]
    (fn check [instance instance-path]
      (when-let [err-props (c instance instance-path)]
        (assert (map? err-props)
                (pr-str [schema-path check-fn err-props]))
        [(-> {:issue                      issue-type
              :instance                       instance
              :in                        instance-path
              :schema-path               schema-path
              (keyword (kebab (name k))) schema-val}
             (merge err-props))]))))

(defn- mk-pred
  [context schema-path pred-fn]
  (mk-check context schema-path
            (fn [schema-val]
              (let [p (pred-fn schema-val)]
                (fn pred-check [instance _]
                  (when-not (p instance)
                     {}))))))

(defn- type-pred
  "Type t is a string or a collection of strings"
  [t]
  (cond
    (string? t)
    (case t
      "string"  string?
      "array"   sequential?
      "boolean" boolean?
      "integer" integer?
      "null"    nil?
      "number"  number?
      "object"  map?)

    (sequential? t)
    (fn [instance]
      (some #(% instance) (map type-pred t)))))

(defmethod key-validator :type
  [context schema-path]
  (mk-pred context schema-path type-pred))

(defmethod key-validator :maxLength
  [context path]
  (mk-pred context path
           (fn [max-length]
             (fn [instance]
               (if (string? instance)
                 (<= (count instance) max-length)
                 true)))))

(defmethod key-validator :minLength
  [context path]
  (mk-pred context path
           (fn [min-length]
             (fn [instance]
               (if (string? instance)
                 (>= (count instance) min-length)
                 true)))))

(defmethod key-validator :pattern
  [context path]
  (mk-pred context path
           (fn [pattern]
             (let [regex (java.util.regex.Pattern/compile pattern)]
               (fn [instance]
                 (if (string? instance)
                   (re-find regex instance)
                   true))))))

(def ^:private uuid-re
  #"^[0-9a-fA-F]{8}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{12}$")

(defmethod key-validator :format
  [context path]
  (mk-pred context path
           (fn [format]
             (case (as-key format)
               ;; TODO: add more formats
               :uuid (fn [s]
                       (if (string? s)
                         (re-find uuid-re s)
                         true))
               ;; default ok
               (fn [_]
                 true)))))

(defmethod key-validator :multipleOf
  [context path]
  (mk-pred context path
           (fn [multiple-of]
             {:pre [(pos? multiple-of)]}
             (fn [instance]
               (when (number? instance)
                 (zero? (rem instance multiple-of)))))))

(defn sibling-path
  [path k]
  (assoc path (dec (count path)) k))

(defmethod key-validator :maximum
  [{:keys [specification] :as context} path]
  (let [exclusive? (-> specification
                       (get-in (sibling-path path :exclusiveMaximum)))
        cmp        (if exclusive? < <=)
        err-props  (if (some? exclusive?)
                     {:exclusive-maximum exclusive?}
                     {})]
    (mk-check context path
              (fn [maximum]
                {:pre [(number? maximum)]}
                (fn [instance _]
                  (when (and (number? instance)
                             (not (cmp instance maximum)))
                    err-props))))))

(defmethod key-validator :minimum
  [{:keys [specification] :as context} path]
  (let [exclusive? (-> specification
                       (get-in (sibling-path path :exclusiveMinimum)))
        cmp        (if exclusive? > >=)
        err-props  (if (some? exclusive?)
                     {:exclusive-minimum exclusive?}
                     {})]
    (mk-check context path
              (fn [minimum]
                {:pre [(number? minimum)]}
                (fn [instance _]
                  (when (and (number? instance)
                             (not (cmp instance minimum)))
                    err-props))))))

;; TODO: Check for impact on "unevaluatedItems"; section 10.3.1.3

(defmethod key-validator :contains
  [{:keys [specification] :as context} schema-path]
  (let [min-contains (-> specification
                         (get-in (sibling-path schema-path :minContains)))
        max-contains (-> specification
                         (get-in (sibling-path schema-path :maxContains)))
        schema-path  (pointer/deref specification schema-path true)
        validator    (schema-validator context schema-path)
        count-valid  (fn [instance path]
                       (when (sequential? instance)
                         (->> instance
                              (map-indexed (fn [i el]
                                             (validator el (conj path i))))
                              (filter nil?)
                              (count))))]
    (if (= 0 min-contains) ;; may be nil, so can't use `zero?`
      (constantly nil)
      (mk-check context schema-path
                (fn [_]
                  (cond
                    (and min-contains max-contains)
                    (fn [instance path]
                      (let [count (count-valid instance path)]
                        (when-not (<= min-contains
                                      count
                                      max-contains)
                          {:min-contains   min-contains
                           :max-contains   max-contains
                           :contains-count count})))

                    max-contains
                    (fn [instance path]
                      (let [count (count-valid instance path)]
                        (when-not (<= count max-contains)
                          {:contains-count count
                           :max-contains   max-contains})))

                    min-contains
                    (fn [instance path]
                      (let [count (count-valid instance path)]
                        (when-not (<= min-contains count)
                          {:contains-count count
                           :min-contains   min-contains})))

                    :else
                    (fn [instance path]
                      (let [count (count-valid instance path)]
                        (when-not (pos? count)
                          {:contains-count count})))))))))

(defmethod key-validator :items
  [{:keys [specification] :as context} schema-path]
  (let [schema-path         (pointer/deref specification schema-path true)
        validator (schema-validator context schema-path)]
    (fn [instance path]
      (when (sequential? instance)
        (loop [res   nil
               i     0
               items instance]
          (if-let [[item & rst] (seq items)]
            (recur (combine-issues res (validator item (conj path i)))
                   (inc i)
                   rst)
            res))))))

(defmethod key-validator :maxItems
  [context schema-path]
  (mk-pred context schema-path
           (fn [max-items]
             (fn [instance]
               (when (sequential? instance)
                 (<= (count instance) max-items))))))

(defmethod key-validator :minItems
  [context schema-path]
  (mk-pred context schema-path
           (fn [min-items]
             (fn [instance]
               (when (sequential? instance)
                 (>= (count instance) min-items))))))

(defmethod key-validator :uniqueItems
  [context schema-path]
  (mk-pred context schema-path
           (fn [unique]
             (if unique
               (fn [instance]
                 (when (sequential? instance)
                   (unique? instance)))
               (constantly nil)))))

(defmethod key-validator :allOf
  [{:keys [specification] :as context} schema-path]
  (let [[schema-path all-of] (pointer/find specification schema-path)]
    (->> (range (count all-of))
         (map #(schema-validator context (conj schema-path %)))
         (checks))))

(defmethod key-validator :oneOf
  [{:keys [specification] :as context} schema-path]
  (let [[schema-path one-of] (pointer/find specification schema-path)
        validators (->> (range (count one-of))
                        (mapv #(schema-validator context (conj schema-path %))))]
    (fn [instance path]
      (let [results  (mapv (fn [validator]
                             (validator instance path))
                           validators)
            ok-count  (count (filter nil? results))]
        (when-not (=  1 ok-count)
          [{:issue       :one-of-error
            :ok-count    ok-count
            :instance    instance
            :one-of      one-of
            :results     results
            :in          path
            :schema-path schema-path}])))))

(defmethod key-validator :anyOf
  [{:keys [specification] :as context} schema-path]
  (let [[schema-path any-of] (pointer/find specification schema-path)
        validators           (->> (range (count any-of))
                                  (mapv #(schema-validator context (conj schema-path %)))
                                  (apply juxt))]
    (fn [instance path]
      (let [results  (validators instance path) ;; returns vector of vector of issues
            ok-count (count (filter nil? results))]
        (when-not (<  0 ok-count)
          [{:issue       :any-of-error
            :instance    instance
            :results     results
            :any-of      any-of
            :ok-count    ok-count
            :in          path
            :schema-path schema-path}])))))

(defmethod key-validator :properties
  [{:keys [specification] :as context} schema-path]
  {:pre [specification context schema-path]}
  (let [[schema-path properties] (pointer/find specification schema-path)
        validators (->> (keys properties)
                        (map #(vector % (schema-validator context (conj schema-path %))))
                        (into {}))]
    (fn properties-check
      [instance path]
      (when (map? instance)
        (reduce-kv (fn [ret prop-key validator]
                     (if-let [[_ v] (find instance prop-key)]
                       (combine-issues ret (validator v (conj path prop-key)))
                       ret))
                   nil
                   validators)))))

(defmethod key-validator :maxProperties
  [context schema-path]
  (mk-check context schema-path
            (fn [max-properties]
              (fn [instance _path]
                (when (map? instance)
                  (let [cnt (count (keys instance))]
                    (when (> cnt max-properties)
                      {:count cnt})))))))


(defmethod key-validator :minProperties
  [context schema-path]
  (mk-check context schema-path
            (fn [min-properties]
              (fn [instance _path]
                (when (map? instance)
                  (let [cnt (count (keys instance))]
                    (when (< cnt min-properties)
                      {:count cnt})))))))

(defmethod key-validator :required
  [context schema-path]
  (mk-check context schema-path
            (fn [required]
              (fn [instance _path]
                (when (map? instance)
                  (when-let [missing (->> required
                                          (remove #(contains? instance (as-key %)))
                                          seq)]
                    {:missing (vec missing)}))))))

(defmethod key-validator :dependentRequired
  [context schema-path]
  (mk-pred context schema-path
           (fn [dependent]
             (fn [instance]
               (if (map? instance)
                 (every? (fn [[k props]]
                           (if (contains? instance k)
                             (every? #(contains? instance (as-key %))
                                     props)
                             true))
                         dependent)
                 true)))))

(defmethod key-validator :enum
  [context schema-path]
  (mk-pred context schema-path
           (fn [enum]
             (into {}
                   (map #(vector % true) enum)))))

(defmethod key-validator :const
  [context schema-path]
  (mk-pred context schema-path
           (fn [const]
             #(= const %))))

(defmethod key-validator :not
  [{:keys [specification] :as context} schema-path]
  (let [[schema-path not-val] (pointer/find specification schema-path)
        validator             (schema-validator context schema-path)]
    (fn [instance path]
      (when-not (validator instance path)
        [{:issue       :not-error
          :instance    instance
          :in          path
          :schema-path schema-path
          :not         not-val}]))))

;; TODO: prefixItems (???)
;; https://datatracker.ietf.org/doc/html/draft-bhutton-json-schema-01#section-10.3.1.1
;; TODO: additionalProperties, patternProperties (???)
;; https://datatracker.ietf.org/doc/html/draft-wright-json-schema-validation-01#section-6.20
;; TODO: if, then, else,
;; https://datatracker.ietf.org/doc/html/draft-bhutton-json-schema-01#section-10.2.2.1
;; TODO: dependentSchemas
