(ns nl.jomco.openapi.v3.validator
  (:require
   [clojure.string :as string]
   [nl.jomco.json-pointer :as pointer]
   [nl.jomco.openapi.v3.coercer :as coercer]
   [nl.jomco.openapi.v3.media-matcher :refer [media-matcher]]
   [nl.jomco.openapi.v3.path-matcher :refer [paths-matcher]]
   [nl.jomco.openapi.v3.util :refer [cached-at! full-name]]
   [nl.jomco.openapi.v3.validator.json-schema-validator :as schema]))

;; WIP

(defn- normalize-uri
  [uri-prefix uri]
  (let [uri (string/replace uri #"^[^:]*://[^/]*" "")
        uri (if (and uri-prefix (string/starts-with? uri uri-prefix))
              (string/replace-first uri uri-prefix "")
              uri)
        uri (if (= \/ (first uri))
              uri
              (str "/" uri))]
    uri))

(defn- parameter-path
  [in n]
  (case in
    "query"
    [:query-params (name n)]

    "path"
    [:path-params (name n)]

    "header"
    [:headers (string/lower-case (name n))]

    "cookie"
    [:cookies (name n)]))

(defn- parameter-validator
  [{:keys [specification] :as context} schema-path]
  (when-let [[schema-path {:keys [name in required schema]}] (pointer/find specification schema-path)]
    (let [schema-validator
          (if schema
            (schema/schema-validator context
                                   (conj schema-path :schema))
            (constantly nil))

          p
          (parameter-path in name)

          coerce
          (coercer/parameter-coercer context schema-path)]
      (fn [request path]
        (let [instance (get-in request p ::not-found)
              path     (into path p)]
          (if (= ::not-found instance)
            (when required
              [{:issue       :required-param-error
                :name        name
                :in          path
                :schema-path schema-path}])
            (let [[instance issues] (coerce instance path)]
              (if (seq issues)
                issues
                (schema-validator instance path)))))))))

(defn- media-type-validator*
  [{:keys [specification] :as context} schema-path]
  (let [[schema-path {:keys [schema]}]
        (pointer/find specification schema-path true)]
    (if schema
      (let [validate-body (schema/schema-validator context (conj schema-path :schema))]
        (fn [request-or-response path]
          ;; TODO: coerce / parse body
          (validate-body (:body request-or-response) (conj path :body))))
      (constantly nil) ;; any request body is ok
      )))

(defn- media-type-validator
  [{:keys [cache] :as context} schema-path]
  (cached-at! cache
              [::media-type-validators schema-path]
              (media-type-validator* context schema-path)))

(defn- request-body-validator
  [{:keys [specification] :as context} schema-path]
  (let [[schema-path {:keys [required content]}]
        (pointer/find specification schema-path true)
        ranges      (keys content)
        match-media (media-matcher ranges)]
    (fn [request path]
      (let [content-type (get-in request [:headers "content-type"])]
        (if-let [range (match-media content-type)]
          ((media-type-validator context (into schema-path [:content range])) request path)
          (when required
            [{:issue       :content-type-error
              :instance    content-type
              :in          (into path  [:headers "content-type"])
              :schema-path schema-path
              :ranges      (mapv full-name ranges)}]))))))

(defn- request-operation-validator*
  [{:keys [specification] :as context} schema-path]
  (let [[params-path parameters]
        (pointer/find specification (conj schema-path :parameters))

        request-body-path
        (pointer/deref specification (conj schema-path :requestBody))]
    (schema/checks
     (into (if request-body-path
             [(request-body-validator context
                                      request-body-path)]
             [])
           (->> (range (count parameters))
                (map #(parameter-validator context (conj params-path %))))))))

(defn- request-operation-validator
  [{:keys [cache] :as context} schema-path]
  (->> (request-operation-validator* context schema-path)
       (cached-at! cache [::operation-validators schema-path])))

(defn request-validator
  [{:keys [uri-prefix specification] :as context}]
  (let [paths   (keys (:paths specification))
        matcher (paths-matcher paths)]
    ;; TODO: normalize requests (method, body, params)
    (fn [{:keys [uri method] :as request} path]
      (let [uri (normalize-uri uri-prefix uri)]
        (if-let [{:keys [template parameters]} (matcher uri)]
          (if-let [op-path (pointer/deref specification [:paths template method])]
            ((request-operation-validator context op-path) (assoc request :path-params parameters) path)
            [{:issue       :methods-error
              :instance    method
              :in          (conj path :method)
              :schema-path [:paths template]
              :methods     (keys (get paths template))}])
          [{:issue       :paths-error
            :instance    uri
            :in          (conj path :uri)
            :schema-path [:paths]
            :paths       paths}])))))

;; for requests, parameter-validator is used to check headers
(defn- response-header-validator
  [{:keys [specification] :as context} schema-path]
  (let [name (full-name (last schema-path))]
    (when-let [[schema-path {:keys [required schema]}]
               (pointer/find specification schema-path)]
      (let [schema-validator
            (if schema
              (schema/schema-validator context
                                     (conj schema-path :schema))
              (constantly nil))

            coerce
            (coercer/header-coercer context schema-path name)]
        (fn [response path]
          (let [instance (get-in response [:headers name] ::not-found)
                path     (into path [:headers name])]
            (if (= ::not-found instance)
              (when required
                [{:issue       :required-param-error
                  :name        name
                  :in          path
                  :schema-path schema-path}])
              (let [[instance issues] (coerce instance path)]
                (if (seq issues)
                  issues
                  (schema-validator instance path))))))))))

(defn- response-object-validator*
  [{:keys [specification] :as context} schema-path]
  (let [[headers-path headers]
        (pointer/find specification
                           (conj schema-path :headers))

        headers-validator
        (when headers
          (->> (keys headers)
               (map #(response-header-validator context (conj headers-path %)))
               (schema/checks)))

        [content-path content]
        (pointer/find specification
                           (conj schema-path :content))

        match-media (media-matcher (keys content))]
    (fn [{{:strs [content-type]} :headers :as response} path]
      (if-let [range (match-media content-type)]
        (cond->
            ((media-type-validator context (conj content-path range))
             response path)
          headers-validator
          (schema/combine-issues (headers-validator response path)))
        [{:issue       :content-type-error
          :instance    content-type
          :in          (into path [:headers "content-type"])
          :schema-path content-path
          :ranges      (mapv full-name (keys content))}]))))

(defn- response-object-validator
  [{:keys [cache] :as context} schema-path]
  (cached-at! cache
              [::response-object-validators schema-path]
              (response-object-validator* context schema-path)))

(defn- status-to-range
  [status]
  (keyword (str (first (str status))
                "XX")))

(defn- status-matcher
  [ranges]
  (let [ranges (set ranges)]
    (->> (range 100 600)
         (keep (fn [status]
                (when-let [range (or (ranges (keyword (str status)))
                                     (ranges (status-to-range status))
                                     (ranges :default))]
                  [status range])))
         (into {}))))

(defn- response-operation-validator*
  [{:keys [specification] :as context} schema-path]
  (let [[responses-path responses]
        (pointer/find specification
                           (conj schema-path :responses)
                           true)

        match-status
        (status-matcher (keys responses))]
    (fn [{:keys [status] :as response} path]
      (if-let [range (match-status status)]
        ((response-object-validator context (conj responses-path range)) response path)
        [{:issue       :status-error
          :instance    status
          :in          (conj path :status)
          :schema-path responses-path}]))))

(defn- response-operation-validator
  [{:keys [cache] :as context} schema-path]
  (cached-at! cache
              [::response-operation-validators schema-path]
              (response-operation-validator* context schema-path)))

(defn response-validator
  [{:keys [uri-prefix specification] :as context}]
  (let [paths   (keys (:paths specification))
        matcher (paths-matcher paths)]
    ;; TODO: normalize requests (method, body, params)
    (fn [{:keys [request response]} path]
      (let [{:keys [uri method]} request
            uri                  (normalize-uri uri-prefix uri)]
        (if-let [{:keys [template]} (matcher uri)]
          (if-let [op-path (pointer/deref specification
                                               [:paths template method])]
            ((response-operation-validator context op-path) response (conj path :response))
            [{:issue       :methods-error
              :instance    method
              :in          (into path [:request :method])
              :schema-path [:paths template]
              :methods     (keys (get paths template))}])
          [{:issue       :paths-error
            :instance    uri
            :in          (into path [:request :uri])
            :schema-path [:paths]
            :paths       paths}])))))

(defn interaction-validator
  [{:keys [uri-prefix specification] :as context}]
  (let [response-val (response-validator context)
        request-val  (request-validator context)
        paths        (keys (:paths specification))
        matcher      (paths-matcher paths)]
    (fn [{:keys [request] :as interaction} path]
      (let [{:keys [uri method]} request
            uri                  (normalize-uri uri-prefix uri)]
        (if-let [{:keys [template]} (matcher uri)]
          (if [(pointer/deref specification
                                   [:paths template method])]
            (schema/combine-issues
             (request-val request (conj path :request))
             (response-val interaction path))
            [{:issue       :methods-error
              :instance    method
              :in          (into path [:request :method])
              :schema-path [:paths template]
              :methods     (keys (get paths template))}])
          [{:issue       :paths-error
            :instance    uri
            :in          (into path [:request :uri])
            :schema-path [:paths]
            :paths       paths}])))))

(defn validator-context
  [specification opts]
  (merge (schema/validator-context specification)
         opts))
