(ns nl.jomco.openapi.v3.coercer
  (:require
   [clojure.string :as string]
   [nl.jomco.json-pointer :as pointer])
  (:import
   (java.util.regex Pattern)))

(defn- split-simple
  [s]
  (string/split s #","))

(defn- simple-exploder
  [s]
  (if (string? s)
    [s]
    s))

(defn default-style
  [in]
  (case in
    "path"   "simple"
    "query"  "form"
    "header" "simple"
    "cookie" "form"))

(defn default-expode
  [in]
  (case in
    "path"   false
    "query"  true
    "header" true
    "cookie" true))

(defn- mk-splitter
  "Return a splitter that converts the input string to a vector of
  strings."
  [in style explode? n]
  (let [style (or style (default-style in))
        explode? (if (some? explode?)
                   explode?
                   (default-expode in))]
    (case style
      "simple"
      split-simple

      "label"
      (let [sep (if explode?  #"," #"\.")]
        (fn [s]
          (when (string/starts-with? s ".")
            (-> s
                (subs 1)
                (string/split s sep)))))

      "matrix"
      (let [sep (if explode?
                  (Pattern/quote (str ";" (name n) "="))
                  #",")]
        (fn [s]
          (when (string/starts-with? s (str ";" (name n) "="))
            (-> s
                (subs (+ 2 (count (name n))))
                (string/split sep)))))

      "form"
      (if explode?
        simple-exploder
        ;; multiple items
        #(string/split % #","))

      "spaceDelimited"
      (if explode?
        simple-exploder
        ;; multiple items
        #(string/split % #" "))

      "pipeDelimited"
      (if explode?
        simple-exploder
        ;; multiple items
        #(string/split % #"\|")))))

(defn- mk-object-parser
  "Return an object parser that converts the input string to a map
  of strings."
  [in style explode? n]
  (let [style    (or style (default-style in))
        explode? (if (some? explode?)
                   explode?
                   (default-expode in))]
    (case style
      "simple"
      (if explode?
        (fn [s]
          (into {} (partition 2 (string/split s #","))))
        (fn [s]
          (into {} (->> (string/split s #",")
                        (map #(string/split % #"="))))))
      "label"
      (if explode?
        (fn [s]
          (into {} (partition 2 (-> s
                                    (string/replace #"^\." "")
                                    (string/split #",")))))
        (fn [s]
          (into {} (->> (string/split s #"\.")
                        next ;; starts with a dot
                        (map #(string/split % #"="))))))

      "matrix"
      (if explode?
        (let [prefix (Pattern/quote (str ";" (name n) "="))]
          (fn [s]
            (into {} (->> (-> s
                              (string/replace prefix "")
                              (string/split #";"))
                          (map #(string/split % #"="))))))
        (fn [s]
          (-> s
              (->> (string/split s #";")
                   next
                   (map #(string/split % #"="))))))

      "form"
      (if explode?
        ;; FIXME -- need access to whole params object
        (throw (ex-info "Can't explode form object" {}))
        (fn [s]
          (into {} (partition 2 (string/split s ",")))))

      ;; TODO: deepObject
      )))

(defn identity-coercer
  [instance]
  [instance nil])

(defn- simple-parser
  "Return a parser for a simple (non-collection) `type`"
  [type]
  (case type
    "integer"
    parse-long

    "number"
    parse-double

    "boolean"
    parse-boolean

    (nil "string")
    identity

    nil))

(defn- mk-simple-coercer
  "Return a coercer for a simple parsable `type`

  otherwise returns nil."
  [type schema-path]
  (when-let [parser (simple-parser type)]
    (fn simple-coercer [instance path]
      (when instance ;; TODO: ??
        (if-let [instance (parser instance)]
          [instance nil]
          [nil [{:issue       :coercion-error
                 :type        type
                 :instance    instance
                 :in          path
                 :schema-path schema-path}]])))))

;; Coercers take an instance and instance path (like a validator) but
;; return a result pair (vector): a parsed instance and a collection
;; of issues.

(defn schema-coercer
  [{:keys [specification]} schema-path]
  (when-let [[schema-path schema] (pointer/find specification
                                                     schema-path)]
    (mk-simple-coercer (:type schema "string") schema-path)))

(defn parameter-coercer
  [{:keys [specification] :as context} schema-path]
  (when-let [[schema-path {:keys [style explode name in]}]
             (pointer/find specification schema-path false)]
    (assert (string? in))
    (let [schema    (-> specification
                        (pointer/get (conj schema-path :schema)))
          root-type (:type schema "string")]
      (or (mk-simple-coercer root-type schema-path)
          (case root-type
            "array"
            (let [split  (mk-splitter in style explode name)
                  coerce (schema-coercer context
                                         (into schema-path
                                               [:schema :items]))]
              (fn array-coercer [instance path]
                (let [items     (split instance)
                      coercions (->> items
                                     (map-indexed
                                      #(coerce %2 (conj path %1))))]
                  [(mapv first coercions)
                   (mapcat second coercions)])))

            "object" ;; TODO - deepObject, spaceDelimited,
                     ;; pipeDelimited and form explode=true
            (let [parser (mk-object-parser in style explode name)
                  coerce (schema-coercer context
                                        (conj schema-path [:schema]))]
              (fn object-coercer [instance path]
                (coerce (parser instance) path))))))))

;; TODO: refactor common code with parameter-coercer
(defn header-coercer ;; for responses
  [{:keys [specification] :as context} schema-path name]
  (when-let [[schema-path {:keys [style explode]}]
             (pointer/find specification schema-path false)]
    (let [schema    (-> specification
                        (pointer/get (conj schema-path :schema)))
          root-type (:type schema "string")]
      (or (mk-simple-coercer root-type schema-path)
          (case root-type
            "array"
            (let [split  (mk-splitter "header" style explode name)
                  coerce (schema-coercer context
                                         (into schema-path
                                               [:schema :items]))]
              (fn array-coercer [instance path]
                (let [items     (split instance)
                      coercions (->> items
                                     (map-indexed
                                      #(coerce %2 (conj path %1))))]
                  [(mapv first coercions)
                   (mapcat second coercions)])))

            "object" ;; TODO - deepObject, spaceDelimited,
            ;; pipeDelimited and form explode=true
            (let [parser (mk-object-parser "header" style explode name)
                  coerce (schema-coercer context
                                         (conj schema-path [:schema]))]
              (fn object-coercer [instance path]
                (coerce (parser instance) path))))))))
