(ns nl.jomco.json-pointer
  "Parse and resolve JSON Pointers"
  (:require [clojure.string :as string])
  (:refer-clojure :exclude [deref get-in get find]))

(defn as-key
  "Coerce `x` to key (keyword or integer)."
  [x]
  (cond (keyword? x)
        x

        (string? x)
        (keyword x)

        (integer? x)
        x

        :else
        (throw (ex-info (str "Can't convert " x " to keyword")
                        {:x x}))))

(defn- keywordize-path
  "Coerce path components to keys. See `as-key`."
  [path]
  (mapv as-key path))

(defn pointer-path
  "Parse a JSON Pointer as a path vector.

  The returned path's keys are integers or keywords.

  See also `as-key`."
  [pointer]
  (cond
    (string/starts-with? pointer "#/")
    (-> pointer
        (subs 2)
        (string/split #"/")
        keywordize-path)
    :else
    (throw (ex-info (str "Can't parse json pointer " pointer)
                    {:pointer pointer}))))

(defn as-path
  "Coerce `x` to a path. If `x` is a string, parse as JSON pointer."
  [x]
  (if (sequential? x)
    x
    (pointer-path x)))

(defn find
  "Dereference `path` and find node in `document`.

  Given a document that may contain $ref pointers, and a path, return
  a tuple (vector) of the dereferenced path and the node at that
  path. Throws an exception when cyclic references are encountered.

  If `path` is a string, it will be parsed as a JSON pointer. The
  returned path is always a vector.

  Returns nil when path does not point to a node in the document. If
  `throw-on-nil?` is provided and true, raise an error instead.

  See also `deref` and `pointer-path`."
  ([document path throw-on-nil?]
   {:pre [(some? document) path]}
   (loop [node      document
          node-path path ;; path from the current node
          root-path path ;; path from the root
          resolved  []
          seen      #{}]
     (cond
       (:$ref node)
       (let [pp (as-path (:$ref node))]
         (if (seen pp)
           (throw (ex-info (str "Cyclic $refs found dereferencing "
                                (pr-str path))
                           {:resolved  resolved
                            :seen      seen
                            :last-seen pp
                            :path      path}))
           (let [path (into pp node-path)]
             (recur document
                    path
                    path
                    resolved
                    (conj seen pp)))))

       (nil? node)
       (when throw-on-nil?
         (throw (ex-info (str "Path " (pr-str root-path) " resolves to nil")
                         {:path      path
                          :resolved  resolved
                          :root-path root-path}))
         nil)

       (empty? node-path)
       [root-path node]

       :else
       (let [k (as-key (first node-path))]
         (recur (clojure.core/get node k)
                (next node-path)
                root-path
                (conj resolved k)
                seen)))))
  ([document path]
   (find document path false)))

(defn deref
  "Dereference `path` in `document`.

  Given a document that may contain $ref pointers, and a path, return
  the equivalent path that contains no pointers. Throws an exception
  when cyclic references are encountered.

  Returns nil when path does not point to a node in the document. If
  `throw-on-nil?` is provided and true, raise an error instead.

  See also `find` and `get`."
  ([document path throw-on-nil?]
   (first (find document path throw-on-nil?)))
  ([document path]
   (deref document path false)))

(defn get
  "Get node at `path` in `document.

  Given a document that may contain $ref pointers, and a path, return
  the equivalent path that contains no pointers. Throws an exception
  when cyclic references are encountered.

  Returns nil when path does not point to a node in the document. If
  `throw-on-nil?` is provided and true, raise an error instead.

  See also `find` and `get`."

  ([document path throw-on-nil?]
   (second (find document path throw-on-nil?)))
  ([document path]
   (get document path false)))
