(ns nl.jomco.envopts
  "# NAME

  envopts - Clojure library for parsing configuration from environment

  # SYNOPSIS

      (ns your-ns.main
        (:require [nl.jomco.envopts :as envopts]))

      (def opt-specs {:port     [\"port nr\" :int :default 80]
                      :hostname [\"hostname to use\" :str]})

      (defn -main [& args]
        (let [[opts errs] (envopts/opts env opt-specs)]
          (when errs
            (.println *err* (envopts/errs-description errs))
            (System/exit 1))
          ... do something with opts))

  ## Opt Specs

  In envopts, you provide all available configuration options in a map
  of opt specs, which maps environment keys to options:

      {\"PORT\"    [\"The port used to listen to requests\" :int :default 80 :in [:http :port]]
       :hostname [\"The hostname to listen to requests\" :str :in [:http :hostname]]
       :threads  [\"The number of threads to start\" :int]
       :auth-uri [\"Authentication server endpoint\" :http]}

  Keys

  The default in envopts is to use `environ.core/env` as the provider
  of the environment, in which case keys will be lisp-style lower-case
  keywords. I.e. the environment variable `\"USER_NAME\"` will be at key
  `:user-name`.

  It's also possible to specify keys as strings, in which case they wil
  be converted to keywords to match the `environ` conversion.

  Opt Specs

  Opt specs can be specified as a vector - the compact version that is
  nice to read and write by hand, or as a map, which is easier to
  manipulate programmatically.

  Spec vectors begin with a description and an optional type, followed
  by any other options:

      {:port [\"The port that will serve web requests\" :int :default 80]}

  Spec maps can contain the following keys:

    - :description - description of the given key
    - :type - a keyword that is registered in `nl.jomco.envopts/parse`
      multimethod.
    - :parser - parser function for expected type, one of the built-in parsers or
      your own (see below). If :parser is provided, it overrides the :type
      option.
    - :default - the default value for the option. If no default is specified, the key
      is *required*.
    - :in - the path to use for the parsed option in the configuration
      options map. The default is to use the key (converted to lower-snake-case
      keyword if necessary) that is used in the specs map.

  ## types

  Since environment variables are always strings, you need a parser to
  convert an environment value to another type. A few types are
  provided for you. See `as-float`, `as-int`, `as-str` and
  `as-http`.

  ## implementing custom parsers

  A parser function must take a string value and return a tuple:

    - `[value]` or `[value nil]`, where `value` is the parsed value, or
    - `[nil err]` if the string value cannot be parsed.

  `err` should be a partial sentence describing the error. It should
  complete the sentence \"xxx is ...\", so a good value for `err` would be
  \"not a valid frobnitz\". Any errs parsing the configuration map are
  aggregated and returned from `opts` for printing with
  `errs-description`."
  (:require [environ.core :as environ]
            [clojure.string :as string])
  (:import java.net.URI))

(defn as-int
  "Parses s as a Long. Returns `[parsed]` or `[nil error]`."
  [s _opt-spec]
  (try [(Long/parseLong s)]
       (catch NumberFormatException _
         [nil "not a valid integer"])))

(defn as-float
  "Parses s as a Double. Returns `[parsed]` or `[nil error]`."
  [s _opt-spec]
  (try [(Double/parseDouble s)]
       (catch NumberFormatException _
         [nil "not a valid floating point number"])))

(defn as-http
  "Parses s as a web address. s must contain the `http` or `https`
  scheme.

  Valid: \"https://jomco.nl/\"
  Invalid: \"mailto:test@example.com\"
  Invalid: \"jomco.nl\"

  Returns `[parsed]` or `[nil error]`."
  [s _opt-spec]
  (try (let [uri (URI/create s)]
         (if (#{"http" "https"} (.getScheme uri))
           [uri]
           [nil "not a valid http or https URI"]))
       (catch IllegalArgumentException _
         [nil "not a valid http or https URI"])))

(defn as-boolean
  "Parses s as a Boolean. Returns `[parsed]` or `[nil error]`."
  [s _opt-spec]
  (case (string/lower-case s)
    ("true" "yes") [true]
    ("false" "no") [false]
    [nil "Not a valid boolean"]))

(defn as-str
  "Parses s as a string. Returns `[s]`"
  [s _opt-spec]
  [s])

(defmulti parse
  "Parses an environment string `s` according to the `opt-spec`
  map. This is the recommended extension point for handling custom
  types.

  The multimethod dispatches on the value of `:type` in
  `opt-spec`.

  Built-in implementations are:

       - :str
       - :int
       - :float
       - :http
       - :boolean"
  (fn [_s opt-spec]
    (:type opt-spec))
  :default :str)

(defmethod parse :str
  [s opt-spec]
  (as-str s opt-spec))

(defmethod parse :int
  [s opt-spec]
  (as-int s opt-spec))

(defmethod parse :float
  [s opt-spec]
  (as-float s opt-spec))

(defmethod parse :http
  [s opt-spec]
  (as-http s opt-spec))

(defmethod parse :boolean
  [s opt-spec]
  (as-boolean s opt-spec))

(defn- parse-opt
  "Parse string s to given type. Use `parser` in opt-spec if given,
  otherwise delegate to `parse` multi"
  [s {:keys [parser] :as opt-spec}]
  (if parser
    (parser s opt-spec)
    (parse s opt-spec)))

(defn- assoc-opt
  [[opts errs :as res] [k in] [v err]]
  (if err
    [nil (assoc errs k err)]
    (if errs ;; if any errors, return no opts
      res
      [(assoc-in opts in v)])))

(defn- collect-opts
  [env spec]
  (reduce-kv (fn [res k {:keys [default in]
                         :or   {in [k]}
                         :as   opt-spec}]
               (assoc-opt res [k in]
                          (if-let [[_ v] (find env k)]
                            (parse-opt v opt-spec)
                            (if (contains? opt-spec :default) ;; default may be specified as nil
                              [default]
                              [nil "missing"]))))
             nil
             spec))

(defn- as-keyword
  [k]
  (-> k
      (name)
      (string/replace #"[_.]" "-")
      (string/lower-case)
      (keyword)))

(defn- parse-specs
  [compact-specs]
  (reduce-kv (fn [specs k v]
               (assoc specs (as-keyword k)
                      (cond
                        (map? v)
                        v

                        (vector? v)
                        (let [[description type & {:as opts}] v]
                          (assoc opts :description description :type type))

                        :else
                        (throw (IllegalArgumentException. (str "Invalid option spec for " k ", must be a map or a vector"))))))
             {}
             compact-specs))

(defn opts
  "Parse options from the environment map given an option spec.

  Takes one or two arguments:

    - `env` - an optional map of keywords to string values. if `env`
      is not provided, uses `environ.core/env` to get the map from the
      environment.
    - `specs` - a map of keys or strings to option descriptions, one for every
      option. This may be a compact description (vector) or a map. See
      also the ns documentation: `nl.jomco.envopts`.

  Returns an `[options errs]` tuple:

    - `options` - when `env` can be parsed into a valid set of options,
      returns `[options nil]`.
    - `errs` - when any option cannot be parsed, of when a required
      option is not provided in `env`, returns `[nil errs]`.

  `errs` can be printed using `errs-description`.

  See also `specs-description`"
  ([env specs]
   (collect-opts env (parse-specs specs)))
  ([specs]
   (opts environ/env specs)))

(defn- as-env-var
  [k]
  (-> k
      (name)
      (string/replace #"-" "_")
      (string/upper-case)))

(defn- format-columns
  [rows]
  (let [max-width (apply max (map #(-> % first count) rows))]
    (->> rows
         (map (fn [[left right]]
                (format (str "%-" max-width "s  %s")
                        left
                        right)))
         (string/join "\n"))))

(defn specs-description
  "Return a multi-line string description of the option specs, suitable
  for printing to the user. See the ns documentation
  in `nl.jomco.envopts` for the format of `specs`"
  [specs]
  (let [specs (parse-specs specs)]
    (->> specs
         keys
         sort
         (map #(vector (as-env-var %) (:description (specs %))))
         (format-columns))))

(defn errs-description
  "Return a multi-line string description of the errors, suitable for
  printing to the user."
  [errs]
  (->> errs
       keys
       sort
       (map #(vector (as-env-var %) (str "is " (errs %))))
       (format-columns)))

(defn opts!
  "Parse configuration from `env` using `opt-specs` or raise ex-info.
  
  Returns configuration map if succesful, throws an informative
  exception if parsing fails. See also `opts`."
  [env opt-specs]
  (let [[config errs] (opts env opt-specs)]
    (when errs
      (throw (ex-info  (str "Error in environment configuration\n"
                            (errs-description errs) "\n"
                            "Available environment vars:\n"
                            (specs-description opt-specs) "\n")
                       {:errs      errs
                        :config    config
                        :opt-specs opt-specs})))
    config))
