(ns nl.jomco.envopts
  "# NAME

  envopts - Clojure library for parsing configuration from environment

  # SYNOPSIS

      (ns your-ns.main
        (:require [nl.jomco.envopts :as envopts]))

      (defn -main [& args]
        (let [[opts errs] (envopts/opts env SPEC)]
          (when errs
            (println (envopts/errs-description errs))
            (System/exit 1))
          ... do something with opts))

  ## specs

  In envopts, you provide all available configuration options in a map
  of /specs/, which maps environment keys to options:

      {:environment-key {:parser  envopts/as-str
                         :default some-value
                         :opt-key option-key}
       :other-key       ...}

  Keys

  The default in envopts is to use [[environ.core/env]] as the provider
  of the environment, in which case keys will be lisp-style lower-case
  keywords. I.e. the environment variable `\"USER_NAME\"` will be at key
  `:user-name`.

  Options

  Options are specified as a map of the following keys:

    - :parser - parser function for expected type, one of the built-in parsers or
      your own (see below). The default is `nl.jomco.envopts/as-str`.
    - :default - the default value for the option. If no default is specified, the key
      is *required*.
    - :opt-key - the key to use for the parsed option in the configuration
      options map. The default is to use the same key as used in the
      specs map.

  ## types

  Since environment variables are always strings, you need a parser to
  convert an environment value into another type. A few types are
  provided for you. See [[as-float]], [[as-int]], [[as-str]] and
  [[as-http]].

  ## implementing custom parsers

  A parser function must take a string value and return a tuple:

    - `[value]` or `[value nil]`, where `value` is the parsed value, or
    - `[nil err]` if the string value cannot be parsed.

  `err` should be a partial sentence describing the error. It should
  complete the sentence \"xxx is ...\", so a good value for `err` would be
  \"not a valid frobnitz\". Any errs parsing the configuration map are
  aggregated and returned from [[opts]] for printing with
  [[errs-description]]."
  (:require [environ.core :as environ]
            [clojure.string :as string])
  (:import java.net.URI))

(defn as-int
  "Parses s as a Long. Returns `[parsed]` or `[nil error]`."
  [s]
  (try [(Long/parseLong s)]
       (catch NumberFormatException _
         [nil "not a valid integer"])))

(defn as-float
  "Parses s as a Double. Returns `[parsed]` or `[nil error]`."
  [s]
  (try [(Double/parseDouble s)]
       (catch NumberFormatException _
         [nil "not a valid floating point number"])))

(defn as-http
  "Parses s as a web address. s must contain the `http` or `https`
  scheme.

  Valid: \"https://jomco.nl/\"
  Invalid: \"mailto:test@example.com\"
  Invalid: \"jomco.nl\"

  Returns `[parsed]` or `[nil error]`."
  [s]
  (try (let [uri (URI/create s)]
         (if (#{"http" "https"} (.getScheme uri))
           [uri]
           [nil "not a valid http or https URI"]))
       (catch IllegalArgumentException _
         [nil "not a valid http or https URI"])))

(defn as-str
  "Parses s as a string. Returns `[s]`"
  [s]
  [s])

(defn- assoc-opt
  [[opts errs :as res] k [v err]]
  (if err
    [nil (assoc errs k err)]
    (if errs ;; if any errors, return no opts
      res
      [(assoc opts k v)])))

(defn- collect-opts
  [env spec]
  (reduce-kv (fn [res k {:keys [default parser opt-key]
                         :or   {parser as-str opt-key k}
                         :as   opt-spec}]
               (assoc-opt res opt-key
                          (if-let [[_ v] (find env k)]
                            (parser v)
                            (if (contains? opt-spec :default) ;; default may be specified as nil
                              [default]
                              [nil "missing"]))))
             nil
             spec))

(defn opts
  "Parse options from the environment map given an option spec.

  Takes one or two arguments:

    - `env` - an optional map of keywords to string values. if `env`
      is not provided, uses [[environ.core/env]] to get the map from the
      environment.
    - `specs` - a map of keys to option descriptions, one for every
      option. The full specification is in the ns
      documentation: [[nl.jomco.envopts]].

  Returns an `[options errs]` tuple:

    - `options` - when `env` can be parsed into a valid set of options,
      returns `[options nil]`.
    - `errs` - when any option cannot be parsed, of when a required
      option is not provided in `env`, returns `[nil errs]`.

  `errs` can be printed using [[errs-description]].

  See also [[specs-description]]"
  ([env specs]
   (collect-opts env specs))
  ([specs]
   (opts environ/env specs)))

(defn- as-env-var
  [k]
  (-> k
      (name)
      (string/replace #"-" "_")
      (string/upper-case)))

(defn- format-columns
  [rows]
  (let [max-width (apply max (map #(-> % first count) rows))]
    (->> rows
         (map (fn [[left right]]
                (format (str "%-" max-width "s  %s")
                        left
                        right)))
         (string/join "\n"))))

(defn specs-description
  "Return a multi-line string description of the option specs, suitable
  for printing to the user."
  [specs]
  (->> specs
       keys
       sort
       (map #(vector (as-env-var %) (:description (specs %))))
       (format-columns)))

(defn errs-description
  "Return a multi-line string description of the errors, suitable for
  printing to the user."
  [errs]
  (->> errs
       keys
       sort
       (map #(vector (as-env-var %) (str "is " (errs %))))
       (format-columns)))
