(ns nl.jomco.envopts
  (:require [environ.core :as environ]
            [clojure.string :as string])
  (:import java.net.URI))

(defn as-int
  [s]
  (try [(Long/parseLong s)]
       (catch NumberFormatException _
         [nil "not a valid integer"])))

(defn as-float
  [s]
  (try [(Double/parseDouble s)]
       (catch NumberFormatException _
         [nil "not a valid floating point number"])))

(defn as-http
  "Parses s as a web address. s must contain the `http` or `https`
  scheme.

  Valid: \"https://jomco.nl/\"
  Invalid: \"mailto:test@example.com\"
  Invalid: \"jomco.nl\""
  [s]
  (try (let [uri (URI/create s)]
         (if (#{"http" "https"} (.getScheme uri))
           [uri]
           [nil "not a valid http or https URI"]))
       (catch IllegalArgumentException _
         [nil "not a valid http or https URI"])))

(defn as-str
  [s]
  [s])

(defn- assoc-opt
  [[opts errs :as res] k [v err]]
  (if err
    [nil (assoc errs k err)]
    (if errs ;; if any errors, return no opts
      res
      [(assoc opts k v)])))

(defn- collect-opts
  [env spec]
  (reduce-kv (fn [res k {:keys [default parser opt-key]
                         :or   {parser as-str opt-key k}
                         :as   opt-spec}]
               (assoc-opt res opt-key
                          (if-let [[_ v] (find env k)]
                            (parser v)
                            (if (contains? opt-spec :default) ;; default may be specified as nil
                              [default]
                              [nil "missing"]))))
             nil
             spec))

(defn opts
  "Parse options from the environment map given an option spec.

  Returns an `[options errs]` tuple.

  If `env` is not provided, uses `environ.core/env`.

  `specs` must be a map of keys to option descriptions, one for every
  option:

    {:environment-key {:parser  envopts/as-str
                       :default some-value
                       :opt-key option-key}
    ...}

  ## :parser

  Parser function for expected type, one of the built-in parsers or
  your own (see below). The default is `nl.jomco.envopts/as-str`.

  ## :default

  The default value for the option. If no default is specified, the key
  is *required*.

  ## :opt-key

  The key used for the option in the resulting option map. Of not
  specified, the key in the `specs` is used.

  # Return value

  Returns an `[options errs]` tuple.

  ## options

  When `env` can be parsed into a valid set of options, returns
  `[options nil]`.

  ## errs

  When any option cannot be parsed, of when a required option is not
  provided in `env`, returns `[nil errs]`. `errs` can be printed
  using [[errs-description]]."
  ([env specs]
   (collect-opts env specs))
  ([specs]
   (opts environ/env specs)))

(defn- as-env-var
  [k]
  (-> k
      (name)
      (string/replace #"-" "_")
      (string/upper-case)))

(defn- format-columns
  [rows]
  (let [max-width (apply max (map #(-> % first count) rows))]
    (->> rows
         (map (fn [[left right]]
                (format (str "%-" max-width "s  %s")
                        left
                        right)))
         (string/join "\n"))))

(defn specs-description
  "Return a multi-line string description of the option specs, suitable
  for printing to the user."
  [specs]
  (->> specs
       keys
       sort
       (map #(vector (as-env-var %) (:description (specs %))))
       (format-columns)))

(defn errs-description
  "Return a multi-line string description of the errors, suitable for
  printing to the user."
  [errs]
  (->> errs
       keys
       sort
       (map #(vector (as-env-var %) (str "is " (errs %))))
       (format-columns)))
