(ns nl.epij.ring.middleware
  (:require [cheshire.core :as json]
            [nl.epij.gcp.gcf.body :as body]
            [nl.epij.gcp.gcf.message :as message]
            [nl.epij.gcp.gcf.log :as log]
            [clojure.string :as str])
  (:import [java.util Base64]))

(defn wrap-pubsub-data
  "Middleware that decodes the Base64 data of the Pubsub message. Also parses the JSON.

  Make sure the body JSON was parsed before, using something like `ring.middleware.json/wrap-json-body`"
  [handler]
  (fn [{:keys [request-method uri body] :as request}]
    (if (get-in body ["message" "data"])
      (let [{:strs [message subscription]} body
            {data-base64  "data"
             message-id   "messageId"
             publish-time "publishTime"} message
            data-json (String. (.decode (Base64/getDecoder) ^String data-base64))
            data      (json/parse-string data-json)
            message   (message/message message-id publish-time data)
            request'  (assoc request :body (body/body message subscription))]
        (log/info (str "Processing HTTP request: " (str/upper-case (name request-method)) " " uri) request)
        (handler request'))
      (do (log/info (str "Processing HTTP request: " (str/upper-case (name request-method)) " " uri) request)
          (handler request)))))

(comment
 (-> ((wrap-pubsub-data identity)
      {:request-method :post, :uri "/command-fan-out", :query-string nil, :headers {"x-appengine-default-version-hostname" "r93aaf13fd1fb03cap-tp.appspot.com", "host" "europe-west2-henry-de-rijk-elektrotechniek.cloudfunctions.net", "user-agent" "APIs-Google; (+https://developers.google.com/webmasters/APIs-Google.html)", "content-type" "application/json", "x-appengine-country" "ZZ", "forwarded" "for=\"74.125.212.149\";proto=https", "connection" "close", "accept" "application/json", "transfer-encoding" "chunked", "x-appengine-https" "on", "x-forwarded-for" "74.125.212.149", "accept-encoding" "gzip,deflate,br", "from" "noreply@google.com", "x-forwarded-proto" "https", "x-appengine-timeout-ms" "599999", "traceparent" "00-7d67cc84eddefb0a7c12f2a1df490e5b-84cbc479e07985c2-01", "x-appengine-request-log-id" "6060fda800ff00ff5029cf5b9df10001677e7239336161663133666431666230336361702d7470000138613030643762663961396565616131393462353336346636623239633563313a31353100010101", "function-execution-id" "7jvzen07dfcg", "x-appengine-user-ip" "74.125.212.149", "x-cloud-trace-context" "7d67cc84eddefb0a7c12f2a1df490e5b/9568957861014898114;o=1"}, :body {"message" {"data" "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", "messageId" "2129884534111950", "message_id" "2129884534111950", "publishTime" "2021-03-28T22:05:24.662Z", "publish_time" "2021-03-28T22:05:24.662Z"}, "subscription" "projects/henry-de-rijk-elektrotechniek/subscriptions/development-products-imported"}, :server-name "europe-west2-henry-de-rijk-elektrotechniek.cloudfunctions.net", :remote-addr "74.125.212.149", :scheme :https})
     :body
     ::body/message
     keys))
