;;;;
;; Copyright © 2019-2020 anywhere.ninja. All rights reserved.
;; The use and distribution terms for this software are covered by the license which
;; can be found in the file license at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the terms
;; of this license. You must not remove this notice, or any other, from this software.
;;;;

(ns ninja.unifier.response
  "A Clojure(Script) library for unified responses."
  (:refer-clojure :exclude [type meta #?(:cljs -meta)])
  #?@(:clj
      [(:require
         [clojure.core :as core])]
      :cljs
      [(:require
         [cljs.core :as core])
       (:require-macros
         [ninja.unifier.response :refer [try-or]])]))

;;;;
;; Defaults
;;;;

(def ^{:doc     "Default http status for unified error response."
       :dynamic true
       :added   "0.3.0"}
  *default-error-http-status*
  500)


(def ^{:doc     "Default http status for unified success response."
       :dynamic true
       :added   "0.3.0"}
  *default-success-http-status*
  200)


(def ^{:doc     "Default http response."
       :dynamic true
       :added   "0.3.0"}
  *default-http-response*
  {:headers {}})


(def ^{:doc     "Map of unified response types associated with http statuses."
       :dynamic true
       :added   "0.3.0"}
  *default-http-statuses*
  {:error        500
   :exception    500
   :unknown      400
   :warning      400
   :unavailable  503
   :interrupted  400
   :incorrect    400
   :unauthorized 401
   :forbidden    403
   :unsupported  405
   :not-found    404
   :conflict     409
   :busy         503
   :success      200
   :created      201
   :deleted      204
   :accepted     202})



;;;;
;; Macro
;;;;

(declare as-response error?)


#?(:clj
   (defn cljs?
     "Checks &env in macro and returns `true` if that cljs env. Otherwise `false`."
     {:added "0.1.0"}
     [env]
     (boolean (:ns env))))


#?(:clj
   (defmacro try-or
     "Extended version of try-catch.
     Usage:
      * (try-or (/ 1 0))                        ;; => nil
      * (try-or (/ 1 0) false)                  ;; => false
      * (try-or (/ 1 0) #(prn (.getMessage %))) ;; => \"Divide by zero\""
     {:added "0.1.0"}
     ([body]
      `(try-or ~body nil))

     ([body else]
      `(try
         ~body
         (catch ~(if-not (cljs? &env) 'Exception :default) error#
           (when-some [else# ~else]
             (cond
               (boolean? else#) else#
               (fn? else#) (else# error#)
               :else else#)))))))


#?(:clj
   (defmacro not-error->
     "This macro is the same as `clojure.core/some->`, but the check is done
     using the predicate `error?` of the `UnifiedResponse` protocol and
     the substitution occurs as in macro `->` (the `thread-first` macro)."
     {:added "0.4.0"}
     [expr & forms]
     (let [g     (gensym)
           steps (map (fn [step]
                        `(let [g# ~g]
                           (if (error? g#) g# (core/-> g# ~step))))
                      forms)]
       `(let [~g ~expr
              ~@(interleave (repeat g) (butlast steps))]
          ~(if (empty? steps)
             g
             (last steps))))))


#?(:clj
   (defmacro not-error->>
     "This macro is the same as `clojure.core/some->>`, but the check is done
     using the predicate `error?` of the `UnifiedResponse` protocol and
     the substitution occurs as in macro `->>` (the `thread-last` macro)."
     {:added "0.4.0"}
     [expr & forms]
     (let [g     (gensym)
           steps (map (fn [step]
                        `(let [g# ~g]
                           (if (error? g#) g# (core/->> g# ~step))))
                      forms)]
       `(let [~g ~expr
              ~@(interleave (repeat g) (butlast steps))]
          ~(if (empty? steps)
             g
             (last steps))))))


#?(:clj
   (defmacro with-default-http-error-status
     "Overrides the default error http status."
     {:added "0.4.0"}
     [default & body]
     `(binding [*default-error-http-status* ~default]
        ~@body)))


#?(:clj
   (defmacro with-default-success-http-status
     "Overrides the default success http status."
     {:added "0.4.0"}
     [default & body]
     `(binding [*default-success-http-status* ~default]
        ~@body)))


#?(:clj
   (defmacro with-default-http-response
     "Overrides the default http response."
     {:added "0.4.0"}
     [default & body]
     `(binding [*default-http-response* ~default]
        ~@body)))


#?(:clj
   (defmacro with-default-http-statuses
     "Overrides the default http statuses."
     {:added "0.4.0"}
     [default & body]
     `(binding [*default-http-statuses* ~default]
        ~@body)))



;;;;
;; Protocols
;;;;

(defprotocol UnifiedResponse
  "UnifiedResponse protocol."
  :extend-via-metadata true
  (-response? [this]
    "Returns `true` if the given value is unified response. Otherwise `false`.")
  (-error? [this]
    "Returns `true` if the given value is unified error response. Otherwise `false`.")
  (-type [this]
    "Returns `type` of unified response.")
  (-meta [this]
    "Returns `meta` of unified response.")
  (-as-map [this]
    "Returns a unified response as a map or `nil` if the given value is not a unified response.")
  (-as-http [this] [this overrides]
    "Returns a unified response as http response or `nil` if the given value is not a unified response."))


;; Extend `Object` and `nil` for compatibility

#?(:clj
   (extend-protocol UnifiedResponse
     Object
     (-response? [_] false)
     (-error? [_] false)
     (-type [_] nil)
     (-meta [_] nil)
     (-as-map [_] nil)
     (-as-http
       ([_] nil)
       ([_ _] nil)))

   :cljs
   (extend-protocol UnifiedResponse
     default
     (-response? [_] false)
     (-error? [_] false)
     (-type [_] nil)
     (-meta [_] nil)
     (-as-map [_] nil)
     (-as-http
       ([_] nil)
       ([_ _] nil))))


(extend-protocol UnifiedResponse
  nil
  (-response? [_] false)
  (-error? [_] false)
  (-type [_] nil)
  (-meta [_] nil)
  (-as-map [_] nil)
  (-as-http
    ([_] nil)
    ([_ _] nil)))



;;;;
;; Helpers
;;;;

(defn response?
  "Returns `true` if the given value is unified response. Otherwise `false`."
  {:added "0.1.0"}
  [x]
  (-response? x))


(defn error?
  "Returns `true` if the given value is unified error response. Otherwise `false`."
  {:added "0.1.0"}
  [x]
  (-error? x))


(defn type
  "Returns `type` of unified response. Otherwise `nil`."
  {:added "0.1.0"}
  [x]
  (-type x))


(defn meta
  "Returns `meta` of unified response. Otherwise `nil`."
  {:added "0.1.0"}
  [x]
  (-meta x))


(defn as-map
  "Returns a unified response as a map. If the given value is not unified response returns `nil`."
  {:added "0.2.0"}
  [x]
  (-as-map x))


(defn as-http
  "Returns a unified response as http response. If the given value is not unified response returns `nil`."
  {:added "0.3.0"}
  ([x]
   (-as-http x))

  ([x overrides]
   (-as-http x overrides)))


(defn to-map
  "Converts unified response to the map."
  {:added "0.4.0"}
  [x]
  (let [x-type (type x)
        x-meta (meta x)]
    (as-response
      {:type x-type
       :data x
       :meta x-meta}
      {:type   x-type
       :error? (error? x)
       :meta   x-meta})))


(defn type->http-status
  "Returns http status by the given unified response type.
  If type is not registered and `error?` is truthy returns `*default-error-http-status*`.
  Otherwise `*default-success-http-status*`."
  {:added "0.4.0"}
  [type error?]
  (or (get *default-http-statuses* type)
      (if error?
        *default-error-http-status*
        *default-success-http-status*)))


(defn to-http
  "Converts unified response to the http response."
  {:added "0.4.0"}
  ([x]
   (let [status (type->http-status (type x) (error? x))]
     (assoc *default-http-response* :status status :body x)))

  ([x overrides]
   (let [status (type->http-status (type x) (error? x))]
     (merge
       (assoc *default-http-response* :status status :body x)
       overrides))))



;;;;
;; Unified responses
;;;;

(defn as-response
  "Returns unified response."
  {:added "0.1.0"}
  [x {:keys [error? type meta]}]
  (with-meta x
    (assoc (core/meta x)
      `-response? (constantly true)
      `-error? (constantly error?)
      `-type (constantly type)
      `-meta (constantly meta)
      `-as-map to-map
      `-as-http to-http)))



;;;;
;; Unified error responses
;;;;

(defn as-warning
  "Returns unified warning response."
  {:added "0.1.0"}
  ([x] (as-warning x (meta x)))
  ([x meta] (as-response x {:type :warning, :error? true, :meta meta})))


(defn as-error
  "Returns unified error response."
  {:added "0.1.0"}
  ([x] (as-error x (meta x)))
  ([x meta] (as-response x {:type :error, :error? true, :meta meta})))


(defn as-exception
  "Returns unified exception response."
  {:added "0.1.0"}
  ([x] (as-exception x (meta x)))
  ([x meta] (as-response x {:type :exception, :error? true, :meta meta})))


(defn as-unavailable
  "Returns unified unavailable response."
  {:added "0.1.0"}
  ([x] (as-unavailable x (meta x)))
  ([x meta] (as-response x {:type :unavailable, :error? true, :meta meta})))


(defn as-interrupted
  "Returns unified interrupted response."
  {:added "0.1.0"}
  ([x] (as-interrupted x (meta x)))
  ([x meta] (as-response x {:type :interrupted, :error? true, :meta meta})))


(defn as-incorrect
  "Returns unified incorrect response."
  {:added "0.1.0"}
  ([x] (as-incorrect x (meta x)))
  ([x meta] (as-response x {:type :incorrect, :error? true, :meta meta})))


(defn as-unauthorized
  "Returns unified unauthorized response."
  {:added "0.1.0"}
  ([x] (as-unauthorized x (meta x)))
  ([x meta] (as-response x {:type :unauthorized, :error? true, :meta meta})))


(defn as-forbidden
  "Returns unified forbidden response."
  {:added "0.1.0"}
  ([x] (as-forbidden x (meta x)))
  ([x meta] (as-response x {:type :forbidden, :error? true, :meta meta})))


(defn as-not-found
  "Returns unified not-found response."
  {:added "0.1.0"}
  ([x] (as-not-found x (meta x)))
  ([x meta] (as-response x {:type :not-found, :error? true, :meta meta})))


(defn as-unsupported
  "Returns unified unsupported response."
  {:added "0.1.0"}
  ([x] (as-unsupported x (meta x)))
  ([x meta] (as-response x {:type :unsupported, :error? true, :meta meta})))


(defn as-conflict
  "Returns unified conflict response."
  {:added "0.1.0"}
  ([x] (as-conflict x (meta x)))
  ([x meta] (as-response x {:type :conflict, :error? true, :meta meta})))


(defn as-busy
  "Returns unified busy response."
  {:added "0.1.0"}
  ([x] (as-busy x (meta x)))
  ([x meta] (as-response x {:type :busy, :error? true, :meta meta})))


(defn as-unknown
  "Returns unified unknown response."
  {:added "0.1.0"}
  ([x] (as-unknown x (meta x)))
  ([x meta] (as-response x {:type :unknown, :error? true, :meta meta})))



;;;;
;; Unified success responses
;;;;

(defn as-success
  "Returns unified success response."
  {:added "0.1.0"}
  ([x] (as-success x (meta x)))
  ([x meta] (as-response x {:type :success, :error? false, :meta meta})))


(defn as-created
  "Returns unified created response."
  {:added "0.1.0"}
  ([x] (as-created x (meta x)))
  ([x meta] (as-response x {:type :created, :error? false, :meta meta})))


(defn as-deleted
  "Returns unified deleted response."
  {:added "0.1.0"}
  ([x] (as-deleted x (meta x)))
  ([x meta] (as-response x {:type :deleted, :error? false, :meta meta})))


(defn as-accepted
  "Returns unified accepted response."
  {:added "0.1.0"}
  ([x] (as-accepted x (meta x)))
  ([x meta] (as-response x {:type :accepted, :error? false, :meta meta})))
