;;;;
;; Copyright © 2019-2020 anywhere.ninja. All rights reserved.
;; The use and distribution terms for this software are covered by the license which
;; can be found in the file license at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by the terms
;; of this license. You must not remove this notice, or any other, from this software.
;;;;

(ns ninja.unifier.response
  "A simple response unifier."
  (:refer-clojure :exclude [type meta #?(:cljs -meta)])
  #?@(:clj
      [(:require
         [clojure.core :as core]
         [ninja.unifier.helpers :refer [try-or]])]
      :cljs
      [(:require
         [cljs.core :as core])
       (:require-macros
         [ninja.unifier.helpers :refer [try-or]])]))

;;;;
;; Protocols
;;;;

(defprotocol UnifiedResponse
  "UnifiedResponse protocol."
  :extend-via-metadata true
  (-response? [this]
    "Returns `true` if the given value is unified response. Otherwise `false`.")
  (-error? [this]
    "Returns `true` if the given value is unified error response. Otherwise `false`.")
  (-type [this]
    "Returns `type` of unified response.")
  (-meta [this]
    "Returns `meta` of unified response.")
  (-as-map [this]
    "Returns a unified response as map. If the given value is not unified response returns `nil`."))



;;;;
;; Helpers
;;;;

(defn response?
  "Returns `true` if the given value is unified response. Otherwise `false`."
  {:added "0.1.0"}
  [x]
  (or (satisfies? UnifiedResponse x)
      (try-or (-response? x) false)))


(defn error?
  "Returns `true` if the given value is unified error response. Otherwise `false`."
  {:added "0.1.0"}
  [x]
  (try-or (-error? x) false))


(defn type
  "Returns `type` of unified response. Otherwise `nil`."
  {:added "0.1.0"}
  [x]
  (try-or (-type x)))


(defn meta
  "Returns `meta` of unified response. Otherwise `nil`."
  {:added "0.1.0"}
  [x]
  (try-or (-meta x)))


(defn as-map
  "Returns a unified response as map. If the given value is not unified response returns `nil`."
  {:added "0.2.0"}
  [x]
  (try-or (-as-map x)))



;;;;
;; Unified responses
;;;;

;; TODO: add derive to `:error` and `:success` responses?

(defn as-response
  "Returns unified response."
  {:added "0.1.0"}
  [x {:keys [error? type meta]}]
  (with-meta x
    (assoc (core/meta x)
           `-response? (constantly true)
           `-error? (constantly error?)
           `-type (constantly type)
           `-meta (constantly meta)
           `-as-map (constantly {:type type, :data x, :meta meta}))))



;;;;
;; Unified error responses
;;;;

(defn as-warning
  "Returns unified warning response."
  {:added "0.1.0"}
  ([x] (as-warning x nil))
  ([x meta] (as-response x {:type :warning, :error? true, :meta meta})))


(defn as-error
  "Returns unified error response."
  {:added "0.1.0"}
  ([x] (as-error x nil))
  ([x meta] (as-response x {:type :error, :error? true, :meta meta})))


(defn as-exception
  "Returns unified exception response."
  {:added "0.1.0"}
  ([x] (as-exception x nil))
  ([x meta] (as-response x {:type :exception, :error? true, :meta meta})))


(defn as-unavailable
  "Returns unified unavailable response."
  {:added "0.1.0"}
  ([x] (as-unavailable x nil))
  ([x meta] (as-response x {:type :unavailable, :error? true, :meta meta})))


(defn as-interrupted
  "Returns unified interrupted response."
  {:added "0.1.0"}
  ([x] (as-interrupted x nil))
  ([x meta] (as-response x {:type :interrupted, :error? true, :meta meta})))


(defn as-incorrect
  "Returns unified incorrect response."
  {:added "0.1.0"}
  ([x] (as-incorrect x nil))
  ([x meta] (as-response x {:type :incorrect, :error? true, :meta meta})))


(defn as-unauthorized
  "Returns unified unauthorized response."
  {:added "0.1.0"}
  ([x] (as-unauthorized x nil))
  ([x meta] (as-response x {:type :unauthorized, :error? true, :meta meta})))


(defn as-forbidden
  "Returns unified forbidden response."
  {:added "0.1.0"}
  ([x] (as-forbidden x nil))
  ([x meta] (as-response x {:type :forbidden, :error? true, :meta meta})))


(defn as-not-found
  "Returns unified not-found response."
  {:added "0.1.0"}
  ([x] (as-not-found x nil))
  ([x meta] (as-response x {:type :not-found, :error? true, :meta meta})))


(defn as-unsupported
  "Returns unified unsupported response."
  {:added "0.1.0"}
  ([x] (as-unsupported x nil))
  ([x meta] (as-response x {:type :unsupported, :error? true, :meta meta})))


(defn as-conflict
  "Returns unified conflict response."
  {:added "0.1.0"}
  ([x] (as-conflict x nil))
  ([x meta] (as-response x {:type :conflict, :error? true, :meta meta})))


(defn as-busy
  "Returns unified busy response."
  {:added "0.1.0"}
  ([x] (as-busy x nil))
  ([x meta] (as-response x {:type :busy, :error? true, :meta meta})))


(defn as-unknown
  "Returns unified unknown response."
  {:added "0.1.0"}
  ([x] (as-unknown x nil))
  ([x meta] (as-response x {:type :unknown, :error? true, :meta meta})))



;;;;
;; Unified success responses
;;;;

(defn as-success
  "Returns unified success response."
  {:added "0.1.0"}
  ([x] (as-success x nil))
  ([x meta] (as-response x {:type :success, :error? false, :meta meta})))


(defn as-created
  "Returns unified created response."
  {:added "0.1.0"}
  ([x] (as-created x nil))
  ([x meta] (as-response x {:type :created, :error? false, :meta meta})))


(defn as-deleted
  "Returns unified deleted response."
  {:added "0.1.0"}
  ([x] (as-deleted x nil))
  ([x meta] (as-response x {:type :deleted, :error? false, :meta meta})))


(defn as-accepted
  "Returns unified accepted response."
  {:added "0.1.0"}
  ([x] (as-accepted x nil))
  ([x meta] (as-response x {:type :accepted, :error? false, :meta meta})))
