(ns better-record.core)

(def pair-keywords-set
  #{@(def default :default)
    @(def transform :transform)
    @(def valid :valid)})

(def shortcut-keywords-map
  {:not-empty [:valid 'seq]
   :not-nil [:valid '(complement nil?)]
   :not-zero [:valid '#(not (or (nil? %) (zero? %)))]
   :trim [:transform 'clojure.string/trim]})

(defn- transform-single-keywords
  [fields]
  (reduce (fn [v x]
            (if (contains? shortcut-keywords-map x)
              (into v (get shortcut-keywords-map x))
              (conj v x)))
          []
          fields))

(defn- separate-lines
  ([fields]
   (separate-lines [] fields))
  ([lines fields]
   (if (empty? fields)
     lines
     (let [[field & remainder] fields]
       (if-not (keyword? field)
         (recur (conj lines [field]) remainder)
         (if (contains? pair-keywords-set field)
           (recur (conj (pop lines)
                        (conj (last lines) field (first remainder)))
                  (rest remainder))
           (throw (IllegalArgumentException.
                    (str field " is not a valid keyword.")))))))))

(defn- build-line-map
  [line]
  (let [[field & tags] line]
    {(keyword field)
     (into {} (map #(apply vector %) (partition 2 tags)))}))

(defn- build-option-map
  [line-map]
  (let [[field restrictions] (first line-map)]
    (reduce-kv (fn [m k v]
                 (assoc m k (merge (get m k) {field v})))
               {}
               restrictions)))

(defn- merge-options
  [target-map option-map]
  (reduce-kv
    (fn [m k v]
      (assoc m k (merge (get m k) v)))
    target-map
    option-map))

(defn- build-record-map
  [lines]
  (let [option-maps (map (comp build-option-map build-line-map) lines)
        basic-map (reduce merge-options {} option-maps)
        basic-defaults (@#'default basic-map)]
    (assoc basic-map @#'default (into {} (map #(vector % (% basic-defaults))
                                              (map (comp keyword first) lines))))))

(defn- apply-transforms
  [record-map transforms-map]
  (reduce-kv (fn [m k v]
               (if (and v (contains? transforms-map k))
                 (assoc m k ((get transforms-map k) v))
                 (assoc m k v)))
             {}
             record-map))

(defn- apply-validations
  [record-map validations-map]
  (map (fn [entry]
         (let [[k v] entry]
           (if (or (not (contains? validations-map k))
                   ((get validations-map k) v))
             entry
             (throw (IllegalArgumentException.
                      (str k " failed validation."))))))
       record-map))

(defmacro create-record-funcs
  [record-type lines]
  (let [fn-name (symbol (str record-type '*))
        gen-fn (symbol (str 'map-> record-type))
        record-map (build-record-map lines)]
    `(do
       (declare ~gen-fn)
       (defn ~fn-name
         ([]
           (~fn-name {}))
         ([basis#]
           (~fn-name basis# {}))
         ([basis# alterations#]
           (-> (merge ~(@#'default record-map) basis# alterations#)
               (apply-transforms ~(@#'transform record-map))
               (apply-validations ~(@#'valid record-map))
               ~gen-fn))))))

(defmacro defrecord*
  [record-type fields & opts+specs]
  (let [adjusted-fields (transform-single-keywords fields)
        lines (separate-lines adjusted-fields)
        field-names (into [] (map first lines))]
    `(do
       (create-record-funcs ~record-type ~lines)
       (defrecord ~record-type ~field-names ~@opts+specs))))
