(ns nicolasoury.updatable-map-reduce.map-reduce
  (:import java.util.concurrent.ConcurrentHashMap 
           java.util.Enumeration
	   [clojure.lang IPersistentCollection
	    ITransientCollection Counted IEditableCollection ILookup
	    IMeta IObj Seqable IPersistentList IPersistentMap IPersistentStack
	    IPersistentSet IPersistentVector
	    Associative
	    ITransientAssociative ITransientMap ITransientVector
	    ITransientSet
            
	    ] 
	   )
  (:gen-class)) 




(set! *warn-on-reflection* true)



(defmacro no-update [] :nicolasoury.updatable-map-reduce.map-reduce/no-update)
(defmacro no-update? [x] `(identical? ~x (no-update)))

(defprotocol Updatable
  (update [x diff-plus diff-minus complete-data]
          "gives an updated result from the diff_plus and the diff_minus. Returns (no-update), if we should not compute the updated result.
complete-data has to be countable.
")
  (get-value [x] "returns the value")
  )


(comment "An Updatee is a structure holding the result of an updatable map reduce.
It has the capability to give an updated result from itself and positive/negative updates. It also evaluates if it is too costly to do so."
         )

 
(defprotocol
  MapReduce
  (map-reduce [x data] "Apply the map reduce x on the data")
  )



(defprotocol
  UpdatableMapReduce
  (update-map-reduce [x old-value diff-plus diff-minus complete-data]
                     "update map-reduction with the diffs. In case inversion was not possible, must return (cannot-invert)"))


(defmacro update-too-expensive [last-read complete-data price-of-computation]
  `(>  ~last-read   100)); (count ~complete-data)))



(defmacro cannot-invert [] :nicolasoury.updatable-map-reduce.map-reduce/cannot-invert)
(defmacro update-ok? [x] `(not (identical? ~x (cannot-invert))))

(deftype Updatee
  [
    ^{:volatile-mutable true} ^int last-read; the number of copies since last-read. Used to evaluate if it is too costly to continue update it
   ^int price-of-computation
   value; the current value
   mr ; the map-reduce used
   ]
  Updatable
  (update [x diff-plus diff-minus complete-data]
        (if (update-too-expensive last-read complete-data price-of-computation)
            ; is it too expensive to update compared to the price of the complete-data
                                        ; it is not too expensive, we update
            (no-update)
            (let [new-val  (update-map-reduce mr value diff-plus diff-minus complete-data)]
              (if (update-ok? new-val)
                (Updatee. (inc last-read) price-of-computation new-val mr)
                (Updatee. (inc last-read) price-of-computation  (map-reduce mr complete-data) mr)))))
  (get-value [x]
             (set! last-read (int 0))
             value))


(defprotocol MapReducable
  (do-map-me-and-reduce [data mr] "use map and reducer mr on data. Can use some particularities of data and mr or call (map-reduce mr data)"))




(comment "The philosophy of a map reduce is the following:
   We have a map function
    map : seq a -> seq b
  We have a reduction function that applies plus and zero:
   zero : c
   plus : c -> b -> c
  In some cases, c can be equal to b, and plus associative/commutative.
  It becomes then possible to use // reductions. We will do that one day in this situation
  To update something, we can
   - add new elements. We use then plus
  Remove elements. We need to have a minus
   - minus : c -> b -> c" )

(defmacro do-map-reduce [initial-code data-code map-code plus-code]
    `(loop [l# ~data-code ~'arg1 ~initial-code]
       (if l#
         (let [~'arg (first l#)
               ~'arg2 ~map-code]
           (recur (next l#) ~plus-code))
         ~'arg1
         )))           
         
(defmacro map-reducer
  "additive-map-reducer creates a map-reducer objects based on addition/substraction of the mapped value.
   The mapped value have to be at least a monoid. (You must provide a :plus and a :zero)
   They can optionaly be a group, (you may provide a :minus)
   You must gives it a few options imcluding:
 - :map defaults to identity, expressed as a term of arg. example: :map (+ arg (int 1))
 - :plus, the resulting term of plus, in tem of arg1 and arg2. example: :plus (assoc arg1 (first arg2) (second arg2))
 - :zero the value of zero, default nil example :zero {}
 - :minus optional, expressed in term of arg1 arg2, default to (cannot-invert)
     example:  :minus  (dissoc arg1 (first arg2))
At some point we will add support for user code to access the next element to reduce.
"
  [& args]
  (let [options-map (apply hash-map args)
        minus-code (or (:minus options-map) `(cannot-invert))
        plus-code (or (:plus options-map) (throw (new Exception "Must give a :plus to a map reduce")))
        map-code (or (:map options-map) 'arg )
        zero-code (:zero options-map)
        ]
    
	`(reify 
          MapReduce
         (map-reduce [~'this ~'data ]  (do-map-reduce ~zero-code (seq ~'data) ~map-code ~plus-code))
         UpdatableMapReduce
         (update-map-reduce [x# current-value# diff-plus# diff-minus# complete-data#]
                            (let [value-plus#  (do-map-reduce current-value# (seq diff-plus#) ~map-code ~plus-code)]
                              (if diff-minus#
                                (let [res#  (do-map-reduce value-plus# (seq diff-minus#) ~map-code ~minus-code)]
                                  (if (update-ok? res#)
                                    res#
                                    (do-map-reduce ~zero-code (seq complete-data#) ~map-code ~plus-code)))
                                value-plus#))))))

(defmacro map-reduce-fn [& options]
  `(let [mr-object# (map-reducer ~@options)]
     (fn [s#] (map-me-and-reduce s# mr-object#))))


(defn map-me-and-reduce [x mr]
  (if (instance? nicolasoury.updatable_map_reduce.map_reduce.MapReducable x)
       (do-map-me-and-reduce x mr)
       (map-reduce mr x)))
  


(comment Most caching mechanism don't like nil. SO we remap nil to another value)

(def remapped-nil :nicolasoury.invertible-map-reduce.map-reduce/nil)

(defmacro map-nil [expr]
  `(let [res# ~expr]
     (if (nil? res#) remapped-nil res#)))

(defmacro unmap-nil [expr]
  `(let [res# ~expr]
     (if (identical? res# remapped-nil) nil res#)))

(defprotocol
  Caching
  (cached [x f] "Returns the cache value of f on x, or nil.")
  (cache [x f v] "Cache v as the value of f. Map nil if necessary"))



(defprotocol
  Diff
  (get-cache [ancestor])
  (update-cache [x cache])
  (force-cache [x])
  )

(declare transient-invertible-map-reducable)
(declare invertible-map-reducable)

(comment "Strategy for computing reduced value:

We use a ConcurrentHashMap in a non-colatile field.
The ConcurrentHashMap goes from map-reducer to the Updatee containing the value for the current MapReducable.
To minimise cost, we only produce a ConcurrentHashMap when the value is actually asked.
 (This allows multiple modifications in a row to be cheap.)
To do so, when someone wants to read/write into the concurrent HashMap:
    a. he check wether there is one, in this case, it reads write freely
    b. if there is nothing (nil), it locks the MapReducable object.
   - Once the lock is acquired, it checks wether there is a ConcurrentHashMap (someone put it in
                                                                                     between)
      unlock, go to a.
    - Else copy and update a ConcurrentHashMap from the ancestor. Put the ancestor to null.
Put the concurrent hash map to the new hash map. unlock. go to a.)
We don't want to have situations where it appears that the map-reducable has no ancestor
nor cache.
For this reason, we put use the same reference for both the ancestor and the cache.
A map-reducable has
  - either the ancestor from which it has been updated
  - either an up to date table
  - In future version, if it is necessary, nil, meaning an empty cache.
To go from one to the other, a lock is necessary.

This strategy is implemented by the following macro."
)


(defmacro small-concurrent-hash-map []
   `(ConcurrentHashMap. (int 8) (float 0.75) (int 2)))

(defmacro acquire-cache
  "check whether there is a concurrent-hash-map, else construct it from ancestor and diff-plus, diff-minus
   Must be used within the deftype of map-reducable"
  [map-reducable cache-name]
  `(if (identical? (type ~cache-name) ConcurrentHashMap) ; we already have a cache
     ~cache-name  
     (locking    ;do ;; should be a locking but that won't work now
     ; (monitor-enter ~map-reducable)
      (if  (identical? (type ~cache-name) ConcurrentHashMap);; maybe someone created it concurrently, or it was done but not in our cache.
        ~cache-name
        (let [new-cache#  (small-concurrent-hash-map)] ;; we have to create the cache
          (update-cache ~map-reducable new-cache#)
          (set! ~cache-name new-cache#)))
      ;(monitor-exit ~map-reducable)
      ~cache-name)))

(comment "We have a corresponding strategy for creating a new map-reducable from
an old one and diff-+ and diff--.
There is two situations, that are equivalent with a small performance difference,
so we don't need any lock:
 - There is a cache in the source map-reducable.
   We take it as an ancestor for the new, using diff-+ and diff--.
 - There is an ancestor in the source ma-reducable. We keep the same ancestor and
   insert the diff-+ and diff-- into the corresponding sets of the source.
   For this, we need a peice of code to extend a diff-+ with another one and a diff--
  with another one. We seq the second one and reduce it into the first one using +.
  In practice, this piece of code is always cons. So we do not ask it.

The name of the current map-reducable must be 'this

This strategy is implemented in the following macro.
"
)

(defmacro updated-map-reducable  [content diff-plus diff-minus]
  `(let [cache# ~'cacheOrAncestor]
     ;; we must look at it only once, to prevent concurrent change
     (if (identical? (type cache#) ConcurrentHashMap)
       (new ~'InvertibleMapReducable
            ~'this ~content ~diff-plus ~diff-minus)
     ;; we need to merge diff-plus and diff-minus
       (let [diffPlus#  (concat ~diff-plus ~'my_positive_diff)
             diffMinus# (concat  ~diff-minus ~'my_negative_diff)]

           ;;  (loop [d# (seq ~diff-plus) ~'arg1 ~'my_positive_diff]
            ;;             (if d#
              ;;             (let [~'arg2 (first d#)]
                ;;             (recur (next d#) (cons ~'arg2 ~'arg1 ))
                  ;;         ~'arg1))

         (new ~'InvertibleMapReducable
              ~'cacheOrAncestor ~content diffPlus# diffMinus#)))))


     

(deftype InvertibleMapReducable
  [
   ^{:unsynchronized-mutable true} cacheOrAncestor
   my_content
   my_positive_diff
   my_negative_diff]
  Caching
  (cached [x f]
          (let [^ConcurrentHashMap cache (get-cache x)]
                (if-let [res (.get cache f)] (unmap-nil res))))
            
  (cache [x f v]
         (let [  ^ConcurrentHashMap cache  (get-cache x)]
               (.put cache  f v)
               v ))
  Diff
  (get-cache [x] (acquire-cache x cacheOrAncestor))
  (update-cache [x cache]
         (let [^ConcurrentHashMap old-cache (get-cache cacheOrAncestor)
               ^Enumeration iter  (.keys old-cache)]
           (loop []
             (when (.hasMoreElements iter)
               (let [key (.nextElement iter)
                     old-val (.get old-cache key) ; val is the old Updatee
                     ]
                 (let [new-val (update old-val my_positive_diff  my_negative_diff my_content)]
                       (when-not (no-update? new-val)
                         (.put  ^ConcurrentHashMap cache key new-val))))
                   (recur))                 
                 )))
  (force-cache [x]
               (get-cache x)
               x)

             
  MapReducable
  (do-map-me-and-reduce [data mr]
                        (if-let [the-val (cached data mr)] ;; the-val is an updatee
                          (get-value the-val)
                          (let [res (map-reduce mr my_content)]
                            (cache data mr (Updatee. (int 0) (int 1) res mr))
                            res)))

  IPersistentCollection
  (empty [_] (invertible-map-reducable (.empty ^IPersistentCollection my_content)))
  (equiv [_ o]
         (if (identical? (type o) InvertibleMapReducable)
           (.equiv ^IPersistentCollection my_content (.my_content  ^InvertibleMapReducable o))
           (.equiv ^IPersistentCollection my_content o))) 

(cons [this x]
	(cond
	 (set? my_content)
           (if (contains? my_content x) this
               (updated-map-reducable (conj my_content x) [x] []))
         (and (map? my_content) (map? x))
            (loop [things (seq x) content my_content positive_diff [] negative_diff []]
              (if things
                (let [[a b] (first things)]
                  (if-let [old (get content a)]
                    (recur (next things)  (assoc content a b)  (conj positive_diff [a b]) (conj negative_diff [a old]))
                    (recur (next things) (assoc content a b) (conj positive_diff [a b])  negative_diff)))
                (updated-map-reducable content positive_diff negative_diff)))


	 (map? my_content)
         (let [[key val] x]
           (updated-map-reducable
            (assoc my_content key val) [[key val]]
            (if-let [old-val (get my_content key)] [[key old-val]] nil)))  
         (vector? my_content)
         (.assocN this (count my_content) x)
         true  ;; this is a list, probably
         (updated-map-reducable (conj my_content x) [x] [])
         ))
  Counted
  (count [x] (count my_content))

  IEditableCollection
  (asTransient [x]
      (let [ancestor cacheOrAncestor]
        (if (identical? (type ancestor) ConcurrentHashMap) ; we have a cache
          (transient-invertible-map-reducable x (transient my_content)
                                              (transient []) (transient []))
          (transient-invertible-map-reducable ancestor (transient my_content)
                                              (transient my_positive_diff)
                                              (transient my_negative_diff)))))

  ILookup
  (valAt [_ key] (.valAt ^ILookup my_content key))
  (valAt [_ key default] (.valAt ^ILookup my_content key default))
  IMeta
  (meta [_] (.meta ^IMeta my_content))

  IObj
  (withMeta [this x] (updated-map-reducable my_content nil nil))
  Seqable
  (seq [_] (seq my_content))
  IPersistentList
  IPersistentStack
  (peek [_](.peek ^IPersistentStack my_content ))
  (pop [this] (let [val (.peek ^IPersistentStack my_content )]
                (updated-map-reducable (.pop  ^IPersistentStack my_content) nil [val])))

  IPersistentMap
   (without [this key]
            (if (contains? my_content key)
              (updated-map-reducable  (.without ^IPersistentMap my_content key) nil  [[key (get my_content key)]])
              this))
   (assocEx [this key val]
            (updated-map-reducable  (.assocEx ^IPersistentMap my_content key val)[[key val]]
                                    (if-let [old-val (get my_content key)] [[key old-val]] nil)))  
  Associative
  (containsKey [_ key] (.containsKey ^Associative my_content key))
  (entryAt [_ key] (.entryAt ^Associative my_content key))
  (assoc [this key val]
    (cond
     (map? my_content)
     (updated-map-reducable  (assoc my_content key val) [[key val]]
                             (if-let [old-val (get my_content key)] [[key old-val]] nil))  

     (vector? my_content)
     (updated-map-reducable  (assoc my_content key val) [val]
                             (if-let [old-val (get my_content key)] [old-val] nil))))  
  IPersistentSet
  (disjoin [this key]
           (if (contains? my_content key)
             (updated-map-reducable  (disj my_content key) nil [val])
             this))
  (contains [this ke] (contains? my_content key))
  (get [this key] (.get ^IPersistentSet my_content key))
  IPersistentVector
  (assocN [this i v]  (let [new-content (.assocN ^IPersistentVector my_content i v)]
			(if (> (count new-content) (count my_content))
                          (updated-map-reducable new-content [v]  nil)
                          (updated-map-reducable new-content [v] [(get my_content i)]))))
  )




(defn invertible-map-reducable
  [content]
     (InvertibleMapReducable. (small-concurrent-hash-map) content nil nil))
 


(deftype TransientInvertibleMapReducable
  [ancestor
   ^{:unsynchronized-mutable true} ^ITransientCollection content
   ^{:unsynchronized-mutable true} ^ITransientVector positive_diff
   ^{:unsynchronized-mutable true} ^ITransientVector negative_diff]
  ITransientCollection
  (conj [this x]
	(cond
	 (instance? ITransientSet content)
	   (if (contains? content x) this  ;; WARNING Clojure is BUGGED
	       (do
		 (set! content (conj! content x))
		 (set! positive_diff (conj! positive_diff x))
		 this))
	 (instance? ITransientMap content)
	     (assoc! this (first x) (second x))
	 (instance? ITransientVector content)
	     (.assocN this (count content) x)
	 true
	      (do
		 (set! content (conj! content x))
		 (set! positive_diff (conj! positive_diff x))
		 this)))
  (persistent [this]
	      (InvertibleMapReducable. ancestor
                                       (persistent! content) (persistent! positive_diff)
                                       (persistent! negative_diff)))
  Counted
  (count [_] (count content))
  ITransientMap
  (without [this key] 
  	   (if-let  [v (get content key)]
	     (do
	       (set! content (dissoc! content key))
	       (set! negative_diff (conj! negative_diff [key v]))
	       )) 
	      this
	      )
  ITransientAssociative
  (assoc [this key val]
    (let [old  (get content key)]
      (cond
       (instance? ITransientMap content)
        (do
	  (set! content (assoc! content key val))
	  (set! positive_diff (conj! positive_diff [key val]))
	  (when old  (set! negative_diff (conj! negative_diff [key old]))))
       (instance? ITransientVector content)
         (do
	  (set! content (assoc! content key val))
	  (set! positive_diff (conj! positive_diff val))
	  (when old  (set! negative_diff (conj! negative_diff old))))
	 ))
    this)
  ITransientVector
  (assocN [this key val] (.assoc ^ITransientAssociative this key val))
  (pop [this]
       (set! negative_diff (conj! negative_diff (get content (- (count content) 1))))
       (set! content (pop! content)))
  ITransientSet
  (disjoin [this key] (when true ;(contains? this key) contains is BUGGY
			(set! content (disj! content key))
			(set! negative_diff (conj! negative_diff key)))
	   this)
  
  (contains [this key] (.contains ^ITransientSet content key))
  (get [this key] (.get ^ITransientSet content key))

  
  )
(defn transient-invertible-map-reducable [i j c x]
  (TransientInvertibleMapReducable. i j c x))


(defmethod print-method InvertibleMapReducable [x y] (print-method (seq x) y))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;; EXAMPLES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def mr-plus (map-reducer :plus (+ arg1 arg2) :minus (- arg1 arg2) :zero 0))
(def a (invertible-map-reducable (range 1 10000000)))
(def b (invertible-map-reducable (range 1 100)))
(def mr-times (map-reducer :plus (* arg1 arg2)
                           :minus (if (zero? arg2) (cannot-invert) (/ arg1 arg2)) :zero 1))


(defn test-plus []
  (time (map-me-and-reduce a mr-plus))
  (time (map-me-and-reduce (conj a 15) mr-plus)))


(defn test-times []
  (time (map-me-and-reduce (conj a 0)  mr-times))
  (time (map-me-and-reduce b mr-times))
 nil
  )


(def updatable-map-reducable-class InvertibleMapReducable)

(def updatable-empty-map (invertible-map-reducable {}))
