// Compiled by ClojureScript 1.9.562 {}
goog.provide('nicheware.platform.utilities.common.math');
goog.require('cljs.core');
/**
 * Rounds a float to the given precision.
 * 
 *   - precision: Number of decimal places to round to. 0 will result in int.
 *   - num: float to be rounded.
 *   - returns: num rounded to precision.
 * 
 * eg:
 * ```clojure
 * (roundn 1.54364 3) => 1.544
 * ```
 */
nicheware.platform.utilities.common.math.roundn = (function nicheware$platform$utilities$common$math$roundn(precision,num){
var factor = Math.pow((10),precision);
return (Math.round((num * factor)) / factor);
});
/**
 * Rounds a value, handling floats and ints and always returning an int.
 * 
 *   - value: Value to be rounded. Can be float or int.
 *   - returns: rounded value as an int.
 */
nicheware.platform.utilities.common.math.round = (function nicheware$platform$utilities$common$math$round(value){
return Math.round(value);
});
/**
 * Ceiling call which converts return value to int to simplify later type checking.
 * 
 *   - value: Value as source for ceiling.
 *   - returns: ceiling of value as an int.
 * 
 * eg:
 * ```clojure
 *   (ceil 1.3) => 2
 * ```
 */
nicheware.platform.utilities.common.math.ceil = (function nicheware$platform$utilities$common$math$ceil(value){
return (Math.ceil(value) | (0));
});
/**
 * Floor call which converts return value to int to simplify later type checking.
 * 
 *   - value: Value as source for floor.
 *   - returns: floor of value as an int.
 * 
 * eg:
 * ```clojure
 *   (floor 1.7) => 1
 * ```
 */
nicheware.platform.utilities.common.math.floor = (function nicheware$platform$utilities$common$math$floor(value){
return (Math.floor(value) | (0));
});
/**
 * Do division in format consistent between ClojureScript (which has no ratio) and Clojure.
 * 
 *   - numer: Numerator for division.
 *   - denom: Denominator for division.
 *   - returns: float of result of division. (java.lang.Double for JVM)
 * 
 * eg:
 * ```clojure
 *   (div 1 2) => 0.5
 * ```
 */
nicheware.platform.utilities.common.math.div = (function nicheware$platform$utilities$common$math$div(numer,denom){
return nicheware.platform.utilities.common.math.roundn.call(null,(4),(numer / denom));
});
/**
 * Absolute difference.
 * 
 *   - values1: First value to be used in difference.
 *   - value2: Second value to be used in difference.
 *   - returns: abs(value2 - value1)
 * 
 * eg:
 * ```clojure
 *   (diff 1 2) => 1
 * ```
 */
nicheware.platform.utilities.common.math.diff = (function nicheware$platform$utilities$common$math$diff(value1,value2){
return Math.abs((value1 - value2));
});
/**
 * Multiply to 4 decimal places.
 * 
 *   - arg1: first arg to be multiplied.
 *   - arg2: second arg to be multiplied.
 *   - returns: arg1 * arg2 with precision of 4.
 * 
 * eg:
 * ```clojure
 *   (mult 1.74 2.887) => 5.0234
 * ```
 */
nicheware.platform.utilities.common.math.mult = (function nicheware$platform$utilities$common$math$mult(arg1,arg2){
return nicheware.platform.utilities.common.math.roundn.call(null,(4),(arg1 * arg2));
});
/**
 * Clamps the change (positive or negative) to the specified absolute magnitude.
 * 
 *   - start: start of difference calculation.
 *   - end: end of difference calculation.
 *   - max-abs-change: Maximum absolute difference to be returned.
 *   - returns: start + max of (end - start) and max-abs-change
 * 
 * eg:
 * ```clojure
 *   (clamp-change 1 9 6) => 7
 *   (clamp-change 9 1 6) => 3
 *   (clamp-change 1 9 10) => 9
 * ```
 */
nicheware.platform.utilities.common.math.clamp_change = (function nicheware$platform$utilities$common$math$clamp_change(start,end,max_abs_change){
var change = (end - start);
if((change < (0))){
return (start + (function (){var x__7745__auto__ = change;
var y__7746__auto__ = ((0) - max_abs_change);
return ((x__7745__auto__ > y__7746__auto__) ? x__7745__auto__ : y__7746__auto__);
})());
} else {
return (start + (function (){var x__7752__auto__ = change;
var y__7753__auto__ = max_abs_change;
return ((x__7752__auto__ < y__7753__auto__) ? x__7752__auto__ : y__7753__auto__);
})());
}
});
/**
 * Return whichever value is the biggest absolute value, retaining the sign on the returned value.
 * 
 * - val1: First input value to compare against.
 * - val2: Second input value to compare against.
 * - returns: either val1 or val2, whichever has biggest absolute value.
 * 
 * eg:
 * ```clojure
 * max-abs(-1 -2) => -2
 * max-abs(1 2)   => 2
 * max-abs(1 -2)  => -2
 * ```
 */
nicheware.platform.utilities.common.math.max_abs = (function nicheware$platform$utilities$common$math$max_abs(val1,val2){
if((Math.abs(val1) > Math.abs(val2))){
return val1;
} else {
return val2;
}
});
/**
 * Creates a function which accepts arguments 0, 1, ....n
 * generating a sequence where each value is a progressive ratio of the supplied total.
 * 
 * eg ratio = 0.5, total 10, then
 * 
 * - initial point is half of total (0.5)
 * - next point is half of remaining distance (7.5)
 * - next is half again of remaining (8.75) etc
 * 
 *   f(x) = f(x-1) + ratio * (total - f(x-1))
 * 
 *   - ratio: float value used to compute next point between current point and total.
 *   - total: end value used to computer intermediate point. No value will be greater than this.
 *   - returns: a function that accepts a single int argument, indicating which nth value is required.
 *  Will compute ratio n times, starting between 0 and total, and then taking result as new starting point.
 */
nicheware.platform.utilities.common.math.make_ratio_sequence_fn = (function nicheware$platform$utilities$common$math$make_ratio_sequence_fn(ratio,total){
var initial_value_17923 = (total * ratio);

return (function (n){
var previous_value = (0);
var current_n = (0);
var target_n = n;
while(true){
var value = (function (){var G__17922 = current_n;
switch (G__17922) {
case (0):
return (0);

break;
case (1):
return (total * ratio);

break;
default:
return (previous_value + (ratio * (total - previous_value)));

}
})();
if(cljs.core._EQ_.call(null,current_n,target_n)){
return value;
} else {
var G__17925 = value;
var G__17926 = (current_n + (1));
var G__17927 = target_n;
previous_value = G__17925;
current_n = G__17926;
target_n = G__17927;
continue;
}
break;
}
});
});
/**
 * Makes a lazy infinite sequence from the ratio function, producing values
 * starting at start and getting closer to end (but never reaching).
 * 
 *   - ratio: float value used to compute next point between current point and end.
 *   - start: initial value in sequence
 *   - end: End value, which will never be reached.
 *   - returns: lazy sequence of points between start and end, approaching end using ratio.
 *  see [[make-ratio-sequence-fn]] for details on function used to produce points.
 * 
 * eg:
 * ```clojure
 * (take 10 (ratio-sequence 0.5 0 20))
 *  =>
 * (0 10.0 15.0 17.5 18.75 19.375 19.6875 19.84375 19.921875 19.9609375)
 * ```
 */
nicheware.platform.utilities.common.math.ratio_sequence = (function nicheware$platform$utilities$common$math$ratio_sequence(ratio,start,end){
var sequence_fn = nicheware.platform.utilities.common.math.make_ratio_sequence_fn.call(null,ratio,(end - start));
return cljs.core.map.call(null,((function (sequence_fn){
return (function (p1__17928_SHARP_){
return (start + sequence_fn.call(null,p1__17928_SHARP_));
});})(sequence_fn))
,cljs.core.range.call(null));
});

//# sourceMappingURL=math.js.map