// Compiled by ClojureScript 1.9.562 {}
goog.provide('nicheware.platform.utilities.common.graphics.line');
goog.require('cljs.core');
goog.require('nicheware.platform.utilities.common.core');
/**
 * Create a function that
 * calculates the co-ordinate [x y, ...] that is fraction between the two given co-ordinates.
 * 
 *   - fraction is a float from 0 to 1.0.
 *   - returns: point co-ords will be floats
 */
nicheware.platform.utilities.common.graphics.line.make_lerp = (function nicheware$platform$utilities$common$graphics$line$make_lerp(point1,point2){
var difference = cljs.core.map.call(null,(function (p1__8712_SHARP_,p2__8711_SHARP_){
return (p2__8711_SHARP_ - p1__8712_SHARP_);
}),point1,point2);
return ((function (difference){
return (function (fraction){
return cljs.core.map.call(null,((function (difference){
return (function (p1__8713_SHARP_,p2__8714_SHARP_){
return (p1__8713_SHARP_ + (p2__8714_SHARP_ * fraction));
});})(difference))
,point1,difference);
});
;})(difference))
});
/**
 * Calculates the point [x y ...] that is fraction between the two given points (of any dimension).
 * 
 * Does multi-dimension vector calculations, not just 2-D points.
 * 
 * - point1: N-dimensional point.
 * - point2: N-dimensional point.
 * - fraction is a float from 0 to 1.0.
 * - returns: point that ios specified fraciont between point1 and point2.  Co-ords will be floats and the same dimension as the incoming points.
 */
nicheware.platform.utilities.common.graphics.line.lerp = (function nicheware$platform$utilities$common$graphics$line$lerp(point1,point2,fraction){
var difference = cljs.core.map.call(null,(function (p1__8716_SHARP_,p2__8715_SHARP_){
return (p2__8715_SHARP_ - p1__8716_SHARP_);
}),point1,point2);
return cljs.core.map.call(null,((function (difference){
return (function (p1__8717_SHARP_,p2__8718_SHARP_){
return (p1__8717_SHARP_ + (p2__8718_SHARP_ * fraction));
});})(difference))
,point1,difference);
});
/**
 * Given two points will create a line equation function.
 * 
 * - start: Vector of start point on the line.  ```[x y]```
 * - end: Vector of end point on the line.  ```[x y]```
 * - returns: function that calculates points on the line. The function will take any X and return the Y corresponding to the point on the line.
 *   The Y result will be a float
 */
nicheware.platform.utilities.common.graphics.line.straight_line_equation = (function nicheware$platform$utilities$common$graphics$line$straight_line_equation(p__8719,p__8720){
var vec__8727 = p__8719;
var x1 = cljs.core.nth.call(null,vec__8727,(0),null);
var y1 = cljs.core.nth.call(null,vec__8727,(1),null);
var start = vec__8727;
var vec__8730 = p__8720;
var x2 = cljs.core.nth.call(null,vec__8730,(0),null);
var y2 = cljs.core.nth.call(null,vec__8730,(1),null);
var end = vec__8730;
var slope = ((y2 - y1) / (x2 - x1));
var constant = (y2 - (slope * x2));
return ((function (slope,constant,vec__8727,x1,y1,start,vec__8730,x2,y2,end){
return (function (x){
return (constant + (slope * x));
});
;})(slope,constant,vec__8727,x1,y1,start,vec__8730,x2,y2,end))
});
/**
 * Creates a function for computing points on a quadratic Bezier curve (from 3 control points)
 * 
 * The returned function will accept a fraction from 0 to 1 which will return
 * a point on the curve running from start to end, controlled by the middle.
 * 
 * The middle point is used to calculate a line between the start and end point.
 * The fraction then computes a point along each of these lines, which will be the
 * tangent to the point on the curve. The point is the fraction of this resulting line.
 * 
 * - start: Start control point. N-dimensional.
 * - middle: Middle control point. N-dimensional.
 * - end: End control point. N-dimensional.
 * - returns: function which takes a single argument from 0 to 1. Returns a N-dimensional point on the curve.
 */
nicheware.platform.utilities.common.graphics.line.bezier_quadratic_equation = (function nicheware$platform$utilities$common$graphics$line$bezier_quadratic_equation(start,middle,end){
var start_lerp_fn = nicheware.platform.utilities.common.graphics.line.make_lerp.call(null,start,middle);
var end_lerp_fn = nicheware.platform.utilities.common.graphics.line.make_lerp.call(null,middle,end);
return ((function (start_lerp_fn,end_lerp_fn){
return (function (fraction){
var start_point = start_lerp_fn.call(null,fraction);
var end_point = end_lerp_fn.call(null,fraction);
return nicheware.platform.utilities.common.graphics.line.lerp.call(null,start_point,end_point,fraction);
});
;})(start_lerp_fn,end_lerp_fn))
});
/**
 * Creates a function for computing points on a quadratic Bezier curve (from 5 control points)
 * The returned function will accept a fraction from 0 to 1 which will return
 * a point on the curve running from start to end, controlled by the middle two control points.
 * 
 * Lines are computed between:
 * 
 * - start -> control1 (L1)
 * - control1 -> control2 (L2)
 * - control2 -> end (L3)
 * 
 * t (fraction 0 to 1) will compute points on each line, resulting in T1, T2, T3
 * 
 * These are then used as a bezier-quadratic-equation.
 * 
 * - start: Start control point. N-dimensional.
 * - control1: Middle control point 1. N-dimensional.
 * - control2: Middle control point 2. N-dimensional.
 * - end: End control point. N-dimensional.
 * - returns: function which takes a single argument from 0 to 1. Returns a N-dimensional point on the curve.
 * 
 * 
 */
nicheware.platform.utilities.common.graphics.line.bezier_cubic_equation = (function nicheware$platform$utilities$common$graphics$line$bezier_cubic_equation(start,control1,control2,end){
var start_lerp_fn = nicheware.platform.utilities.common.graphics.line.make_lerp.call(null,start,control1);
var middle_lerp_fn = nicheware.platform.utilities.common.graphics.line.make_lerp.call(null,control1,control2);
var end_lerp_fn = nicheware.platform.utilities.common.graphics.line.make_lerp.call(null,control2,end);
return ((function (start_lerp_fn,middle_lerp_fn,end_lerp_fn){
return (function (fraction){
var start_point = start_lerp_fn.call(null,fraction);
var middle_point = middle_lerp_fn.call(null,fraction);
var end_point = end_lerp_fn.call(null,fraction);
var bezier_fn = nicheware.platform.utilities.common.graphics.line.bezier_quadratic_equation.call(null,start_point,middle_point,end_point);
return bezier_fn.call(null,fraction);
});
;})(start_lerp_fn,middle_lerp_fn,end_lerp_fn))
});
/**
 * Creates a function to return a Y value given an X for points on a curve
 *   where the curve is represented by the given set of [x y] points
 * 
 *   Finds the two points either size of the given X and then finds points on line between the known points.
 * 
 *   - sample-points: Collection of points (each point is vector ```[x y]```) which are all on a curve.
 *   - returns: function that takes single X argument and returns the correponding Y point on the curve defined by the sample-points.
 *  Function Returns nil if given X is outside the min and max X of the sample points.
 */
nicheware.platform.utilities.common.graphics.line.make_curve_fn_from_samples = (function nicheware$platform$utilities$common$graphics$line$make_curve_fn_from_samples(sample_points){
return (function (x){
var first_x = cljs.core.first.call(null,cljs.core.first.call(null,sample_points));
var last_x = cljs.core.first.call(null,cljs.core.last.call(null,sample_points));
if(((x < first_x)) || ((x > last_x))){
return null;
} else {
var x_limit_fn = ((function (first_x,last_x){
return (function (p__8743,x__$1){
var vec__8744 = p__8743;
var px = cljs.core.nth.call(null,vec__8744,(0),null);
var py = cljs.core.nth.call(null,vec__8744,(1),null);
return (px >= x__$1);
});})(first_x,last_x))
;
var end_index = nicheware.platform.utilities.common.core.find_index.call(null,sample_points,x,x_limit_fn);
var vec__8740 = cljs.core.nth.call(null,sample_points,end_index);
var end_x = cljs.core.nth.call(null,vec__8740,(0),null);
var end_y = cljs.core.nth.call(null,vec__8740,(1),null);
var end_point = vec__8740;
if(cljs.core._EQ_.call(null,end_x,x)){
return end_y;
} else {
var start_point = cljs.core.nth.call(null,sample_points,(end_index - (1)));
var line_fn = nicheware.platform.utilities.common.graphics.line.straight_line_equation.call(null,start_point,end_point);
return line_fn.call(null,x);
}
}
});
});
/**
 * Rasterizes the bezier curve defined by the given start, end and middle control points.
 * 
 *   Will return a value for each int x between the start x and end x.
 * 
 *   As curve points can extend beyond the start and end point, it allows for an optional x-start and x-end to be specified for the rasterized curve, to generate a wider range of points.
 * 
 *   These default to the start and end x.
 * 
 *   - start: Start point defining bezier curve. ```[x y]```
 *   - end: End point defining bezier curve. ```[x y]```
 *   - middle: Middle control point defining bezier curve. ```[x y]```
 *   - x-start: Optional starting X for generated points. Defaults to x of start.
 *   - x-end: Optional ending X for generated points. Defaults to x of end.
 *   - returns: lazy sequence of points, one for each x from x-start to x-end. ```[x y]```
 */
nicheware.platform.utilities.common.graphics.line.rasterize_bezier_quadratic = (function nicheware$platform$utilities$common$graphics$line$rasterize_bezier_quadratic(var_args){
var args8747 = [];
var len__8628__auto___8770 = arguments.length;
var i__8629__auto___8771 = (0);
while(true){
if((i__8629__auto___8771 < len__8628__auto___8770)){
args8747.push((arguments[i__8629__auto___8771]));

var G__8772 = (i__8629__auto___8771 + (1));
i__8629__auto___8771 = G__8772;
continue;
} else {
}
break;
}

var G__8749 = args8747.length;
switch (G__8749) {
case 3:
return nicheware.platform.utilities.common.graphics.line.rasterize_bezier_quadratic.cljs$core$IFn$_invoke$arity$3((arguments[(0)]),(arguments[(1)]),(arguments[(2)]));

break;
case 5:
return nicheware.platform.utilities.common.graphics.line.rasterize_bezier_quadratic.cljs$core$IFn$_invoke$arity$5((arguments[(0)]),(arguments[(1)]),(arguments[(2)]),(arguments[(3)]),(arguments[(4)]));

break;
default:
throw (new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "),cljs.core.str.cljs$core$IFn$_invoke$arity$1(args8747.length)].join('')));

}
});

nicheware.platform.utilities.common.graphics.line.rasterize_bezier_quadratic.cljs$core$IFn$_invoke$arity$3 = (function (p__8750,p__8751,middle){
var vec__8752 = p__8750;
var x1 = cljs.core.nth.call(null,vec__8752,(0),null);
var y1 = cljs.core.nth.call(null,vec__8752,(1),null);
var start = vec__8752;
var vec__8755 = p__8751;
var x2 = cljs.core.nth.call(null,vec__8755,(0),null);
var y2 = cljs.core.nth.call(null,vec__8755,(1),null);
var end = vec__8755;
return nicheware.platform.utilities.common.graphics.line.rasterize_bezier_quadratic.call(null,start,end,middle,x1,x2);
});

nicheware.platform.utilities.common.graphics.line.rasterize_bezier_quadratic.cljs$core$IFn$_invoke$arity$5 = (function (p__8758,p__8759,p__8760,x_start,x_end){
var vec__8761 = p__8758;
var x1 = cljs.core.nth.call(null,vec__8761,(0),null);
var y1 = cljs.core.nth.call(null,vec__8761,(1),null);
var start = vec__8761;
var vec__8764 = p__8759;
var x2 = cljs.core.nth.call(null,vec__8764,(0),null);
var y2 = cljs.core.nth.call(null,vec__8764,(1),null);
var end = vec__8764;
var vec__8767 = p__8760;
var x3 = cljs.core.nth.call(null,vec__8767,(0),null);
var y3 = cljs.core.nth.call(null,vec__8767,(1),null);
var middle = vec__8767;
var bezier_fn = nicheware.platform.utilities.common.graphics.line.bezier_quadratic_equation.call(null,start,middle,end);
var oversample_multiplier = (2);
var t_step = ((1) / ((x_end - x_start) * oversample_multiplier));
var t_range = cljs.core.concat.call(null,cljs.core.range.call(null,(0),(1),t_step),new cljs.core.PersistentVector(null, 1, 5, cljs.core.PersistentVector.EMPTY_NODE, [(1)], null));
var sample_points = cljs.core.map.call(null,bezier_fn,t_range);
var x_to_y_fn = nicheware.platform.utilities.common.graphics.line.make_curve_fn_from_samples.call(null,sample_points);
var points_fn = ((function (bezier_fn,oversample_multiplier,t_step,t_range,sample_points,x_to_y_fn,vec__8761,x1,y1,start,vec__8764,x2,y2,end,vec__8767,x3,y3,middle){
return (function (x){
return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [x,Math.round(x_to_y_fn.call(null,x))], null);
});})(bezier_fn,oversample_multiplier,t_step,t_range,sample_points,x_to_y_fn,vec__8761,x1,y1,start,vec__8764,x2,y2,end,vec__8767,x3,y3,middle))
;
var points = cljs.core.map.call(null,points_fn,cljs.core.range.call(null,x_start,(x_end + (1))));
return points;
});

nicheware.platform.utilities.common.graphics.line.rasterize_bezier_quadratic.cljs$lang$maxFixedArity = 5;

/**
 * Compute the given number of points between start and end.
 * This will divide the curve into points + 1 sections, and gives the curve value
 * for n points.
 * 
 * - interpolation-fn:   Curve interpolation function which should accept t, as value form 0 to 1
 * - points:             An integer number of points to interpolate evenly using the function.
 *  
 */
nicheware.platform.utilities.common.graphics.line.interpolate_n_points = (function nicheware$platform$utilities$common$graphics$line$interpolate_n_points(interpolation_fn,points){
var step_size = ((1) / (points + (1)));
var t_points = cljs.core.map.call(null,((function (step_size){
return (function (p1__8774_SHARP_){
return (step_size * p1__8774_SHARP_);
});})(step_size))
,cljs.core.range.call(null,(1),(points + (1))));
return cljs.core.map.call(null,interpolation_fn,t_points);
});

//# sourceMappingURL=line.js.map