;   Copyright (c) Walter van der Laan, 2010. All rights reserved.

;   The use and distribution terms for this software are covered by the
;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;   which can be found in the file epl-v10.html at the root of this 
;   distribution.
;   By using this software in any fashion, you are agreeing to be bound by
;   the terms of this license.
;   You must not remove this notice, or any other, from this software.

(ns net.wvdlaan.xls
  (:import
   (java.io FileInputStream FileOutputStream)
   (org.apache.poi.hssf.usermodel HSSFWorkbook)
   (org.apache.poi.ss.usermodel Cell DateUtil Sheet Workbook)
   (org.apache.poi.ss.util CellReference)
   (org.apache.poi.xssf.usermodel XSSFWorkbook)))

(defn- book-factory
  "Return a function that will create a POI workbook object."
  [^String fname]
  (if (.endsWith fname ".xlsx")
    (fn
      ([] (XSSFWorkbook.))
      ([^String p] (with-open [inp (FileInputStream. p)]
	     (XSSFWorkbook. inp))))
    (fn
      ([] (HSSFWorkbook.))
      ([^String p] (with-open [inp (FileInputStream. p)]
	     (HSSFWorkbook. inp))))))

(defn- open-book
  "Returns a HSSF or XSSF POI workbook object.
  (open-book \"file.xls\" :existing) => poi-Workbook-object
  (open-book \"file.xlsx\" :either) => poi-Workbook-object
  :existing throws an error if the file does not exist
  :either returns an empty workbook if the file does not exist
  :new always returns an empty workbook, even if the file exists"
  [^String fname status]
  (let [mk-book (book-factory fname)]
    (case status
	  :new      (mk-book)
	  :existing (mk-book fname)
	  :either   (try (mk-book fname) (catch Exception _ (mk-book))))))

(defn- delete-sheet
  "Delete sheet from workbook, eg;
   (delete-sheet (book \"file.xls\") \"sheet1\") => nil"
  [^Workbook book ^String sname]
  (try
    (. book removeSheetAt (. book getSheetIndex sname))
    (catch Exception _ nil)))

;; Date type

(defn excel-date [^Date j]
  "Convert Java Date j to an Excel date. Return nil on errors."
  (let [e (DateUtil/getExcelDate j)]
    (if (< e 0.1) nil e)))

(defn java-date
  "Convert Excel date e to java.lang.Date. Return nil on errors."
  [^double e]
  (DateUtil/getJavaDate e))

;; poi-read

(defn- get-ref
  "Returns the reference of cell c, eg; [0 0] for cell \"A1\"."
  [^Cell c]
  [ (.getRowIndex c) (.getColumnIndex c) ])

(defn- get-cached-value
  "Read the cached primitive value of cell c.
  The returned value can  be a Double, String, Boolean or nil.
  Cells with an error value return nil."
  [^Cell c]
  (let [t1 (.getCellType c)
	t2 (if (= t1 Cell/CELL_TYPE_FORMULA)
	     (.getCachedFormulaResultType c)
	     t1)]
    (cond
     (= t2 Cell/CELL_TYPE_NUMERIC) (.getNumericCellValue c)
     (= t2 Cell/CELL_TYPE_STRING)  (.getStringCellValue c)
     (= t2 Cell/CELL_TYPE_BOOLEAN) (.getBooleanCellValue c)
     :else nil)))

(defn- get-sheet
  "Read sheet at index idx and return a worksheet data structure."
  [^Workbook book idx]
  (let
      [row-iter (iterator-seq (.iterator (.getSheetAt book idx)))]
    (reduce merge
	    (map (fn [cell-iter]
		   (reduce merge
			   (map (fn [c]
				  { (get-ref c) (get-cached-value c) } )
				cell-iter)))
		 row-iter))))

(defn poi-read
  "Read workbook fname into spreadsheet data structure.
  (poi-read \"file.xls\") read all worksheets
  (poi-read \"file.xls\" #{\"sheet1\" \"sheet2\"}) read selected sheets"
  ([fname]
     (poi-read fname (fn [sname] true)))
  ([^String fname pred]
     (let [^Workbook book (open-book fname :existing)]
       (reduce
	(fn [m idx]
	  (let [sname (. book getSheetName idx)]
	    (if (pred sname)
	      (assoc m sname (get-sheet book idx))
	      m)))
	{}
	(range (. book getNumberOfSheets))))))

;; poi-write

(defn- excel-value
  "Convert a clojure value into a primitive value that
  can be used for an excel cell, i.e: Double, Boolean or String."
  [v]
  (cond
   (number? v) (double v)
   (keyword? v) (name v)
   (= (class v) Boolean) v
   (= (class v) java.util.Date) (excel-date v)
   :else (str v)))

(defn- set-value
  "Set the value of cell c to primitive value v."
  [^Cell c v]
  (try
    (. c setCellValue (excel-value v))
    (catch Exception _
      (. c setCellValue (str v)))))

(defn- set-format [^Cell c ^String s ^Workbook book]
  (let [cs (.createCellStyle book)
	df (.createDataFormat book)]
    (. cs setDataFormat (. df getFormat s))
    (. c setCellStyle cs)))

(defn- get-cell
  "Return the POI Cell object for row and col in sheet s.
  Create a new Cell object if it doesn't exist."
  [^Sheet s [row col]]
  (let [r (if-let [r (. s getRow row)] r (. s createRow row))
	c (. r getCell col)]
    (or c (. r createCell col))))

(defn- set-sheet
  "Insert sheet data sdata into sheet sname in workbook book."
  [^String sname sdata slevel ^Workbook book frmts]
  (let [s (if (= slevel :sheet)
	    (do
	      (delete-sheet book sname)
	      (. book createSheet sname))
	    (if-let [x (. book getSheet sname)]
	      x
	      (. book createSheet sname)))]
    (doseq [[cref v] sdata]
      (when v
	(let [c (get-cell s cref)]
	  (set-value c v)
	  (if-let [frmt (get frmts (cref 1))]
	    (if (< 0 (cref 0))
	      (set-format c frmt book))))))))

(defn poi-write
  "Write spreadsheet data structure sdata into workbook fname.
  (poi-write sdata \"file.xls\") uses default slevel; :sheet
  (poi-write sdata \"file.xls\" slevel) slevel is the save level
  The save level can be :book :sheet :value."
  ([bdata fname]
     (poi-write bdata fname :sheet))
  ([bdata ^String fname slevel]
     (poi-write bdata fname slevel {}))
  ([bdata ^String fname slevel frmts]
     (let [^Workbook book (if (= slevel :book)
			    (open-book fname :new)
			    (open-book fname :either))]
       (doseq [sname (keys bdata)]
	 (set-sheet sname (bdata sname) slevel book (get frmts sname [])))
       (with-open [out (FileOutputStream. fname)]
	 (. book write out)))))

;; cell reference

(defn- sref2vref
  "Convert an A1-type cell reference into a [row col] type reference."
  [^String s]
  (let [cr (CellReference. s)]
    [(.getRow cr) (int (.getCol cr))]))

(defn- vref2sref
  "Convert a [row col] type cell reference into an A1-type reference."
  ([[^int r ^int c]]
     (let [cr (CellReference. r c false false)]
       (.formatAsString cr)))
  ([[^int r ^int c] ^String s]
     (let [cr (CellReference. s r c false false)]
       (.formatAsString cr))))

(defn sheetname
  "Return the sheetname part of a A1-type cell reference
  or nil if not present
  (sheetname \"'some sheet'!AD12\") => \"some sheet\""
  [^String s]
  (.getSheetName (CellReference. s)))

(defn offset
  "Return a [row col] type cell reference given an origin and offset.
  (offset \"C4\" [-1 2]) => [2 4]
  (offset [3 2] [-1 2]) => [2 4]
  (offset \"A1\") => [0 0]
  (offset \"C4\") => [3 2]"
  ([c]
     (if (string? c) (sref2vref c) c))
  ([c o]
     (let [[cy cx] (offset c)
	   [oy ox] (offset o)]
       [(+ cy oy) (+ cx ox)])))

(defn excel-ref
  "Return an A1-type cell reference."
  ([c]
     (if (vector? c) (vref2sref c) c))
  ([c s]
     (vref2sref (offset c) s)))

;; sheet to table & table to sheet

(defn header2keyword
  "Convert string s into a valid column keyword."
  [_ ^String s]
  (keyword (apply str (re-seq #"[\w\d-]" (.toLowerCase s)))))

(defn col-num2keyword
  "Convert index i into a valid column keyword."
  [i _]
  (keyword (str "column" i)))

(defn- get-row
  "Return a vector of values taken from sheet sdata.
  Start at cell-ref org and continue in direction dir
  while function pred is true."
  ([sdata org dir]
     (get-row sdata org dir 0 (fn [cnt v] (nil? v))))
  ([sdata org dir nr]
     (get-row sdata org dir nr (fn [cnt v] (< cnt 1))))
  ([sdata org dir nr pred]
     (loop [c (offset org) result [] cnt nr]
       (let [v (sdata c)]
	 (if (pred cnt v)
	   result
	   (recur (offset c dir) (conj result v) (dec cnt)))))))

(defn table
  "Convert, part of, sheet-data sdata into a
  table data structure.
  Start at cell-ref org, or \"A1\" if missing.
  if f is a function it is applied to the first data row to
  create column keywords, otherwise f is assumed to be a
  vector of column keywords and the first row is assumed to
  be a data row."
  ([sdata]
     (table sdata [0 0]))
  ([sdata org]
     (table sdata org header2keyword))
  ([sdata org f]
     (let [org (offset org)
	   hdr (if (fn? f)
		 (vec (map-indexed
		       #(f %1 %2)
		       (get-row sdata org [0 1])))
		 f)
	   cnt (count hdr)]
       (loop [ref (if (fn? f) (offset org [1 0]) org) result []]
	 (let [row (get-row sdata ref [0 1] cnt)]
	   (if (some #(not (nil? %)) row)
	     (recur (offset ref [1 0])
		    (conj result
			  (assoc (zipmap hdr row)
			    :$row
			    (inc (ref 0)))))
	     result))))))

(defn- hdrs [forder sdata]
  (reduce (fn [m [k v]] (assoc m [0 v] k)) sdata forder))

(defn sheet
  "(sheet '({:a 3 :b 6}) [:a])
  => {[0 0] \"a\", [0 1] \"b\",
      [1 1] 6,     [1 0] 3}"
  ([tdata]
     (sheet tdata []))
  ([tdata forder]
     (loop [forder (zipmap forder (iterate inc 0))
	    result (hdrs forder {})
	    row 1
	    tdata tdata]
       (if-let [rec (first tdata)]
	 (let [newcols (zipmap (filter (fn [k]
					 (not (contains? forder k)))
				       (keys rec))
			       (iterate inc (count forder)))
	       result (hdrs newcols result)
	       forder (into forder newcols)]
	   (recur forder
		  (reduce (fn [m [k v]]
			    (assoc m [row (forder k)] v))
			  result
			  rec)
		  (inc row)
		  (rest tdata)))
	 result))))
