;   Copyright (c) Walter van der Laan, 2010. All rights reserved.

;   The use and distribution terms for this software are covered by the
;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;   which can be found in the file epl-v10.html at the root of this 
;   distribution.
;   By using this software in any fashion, you are agreeing to be bound by
;   the terms of this license.
;   You must not remove this notice, or any other, from this software.

(ns net.wvdlaan.table
  (:refer-clojure :exclude [select-keys])
  (:use [clojure.set :only [rename-keys]]))

(defn- select-keys
  "Select elements from map m for which pred is true."
  [m pred]
  (reduce (fn [r [k v]] (if (pred k) (assoc r k v) r)) {} m))

(defn as
  "Rename columns in table tbl as specified by key-map kmap."
  [kmap tbl]
  (map (fn [rec] (rename-keys rec kmap)) tbl))

(defn select
  "Select columns from table tbl for which pred is true."
  [pred tbl]
  (map (fn [rec] (select-keys rec (fn [k] (pred k)))) tbl))

(defn deselect
  "Select columns from table tbl for which pred is false."
  [pred tbl]
  (map (fn [rec] (select-keys rec (fn [k] (not (pred k))))) tbl))

(defn from
  "Returns the cartesian product of two or more tables."
  ([tbl1]
     tbl1)
  ([tbl1 tbl2]
     (for [rec1 tbl1 rec2 tbl2] (merge rec1 rec2)))
  ([tbl1 tbl2 & more]
     (from (from tbl1 tbl2) more)))

(defn keys-to-vals
  "Returns a vector of values for columns ks in record rec."
  [rec ks]
  (vec (map rec ks)))

(defn index
  "Returns an index on table tbl for columns ks."
  [ks tbl]
  (group-by (fn [rec] (keys-to-vals rec ks)) tbl))

(defn order-by
  "Sort table tbl by columns ks."
  [ks tbl]
  (sort-by (fn [rec]
	     (keys-to-vals rec ks))
	   tbl))

(defn unique
  "Returns a function that drops records when lookup does
  not find a match in the lookup-table.
  Use like this: (lookup ks1 ks2 tbl2 (unique f) tbl1)"
  [f]
  (fn [rec1 recs2 k1 k2 v]
    (if (zero? (count recs2)) nil (f rec1 (first recs2)))))

(defn lookup
  "Returns table tbl1 extended with information from lookup-table tbl2.
  Matches columns ks1 of tbl1 with columns ks2 of tbl2.
  Function f is called for each record in tbl1 with zero or more matching
  records from tbl2.
  Function f signature is: (fn [rec1 recs2] ..) -> rec1"
  [ks1 ks2 tbl2 f tbl1]
  (let [idx2 (index ks2 tbl2)]
    (reduce
     (fn [res rec1]
       (let [kv1 (keys-to-vals rec1 ks1)]
	 (if-let [ok-rec (f rec1 (idx2 kv1) ks1 ks2 kv1)]
	   (conj res ok-rec)
	   res)))
     []
     tbl1)))

(defn group
  "Group table tbl by unique values for columns ks
  Function f is called for each group in tbl to perform
  the aggregation.
  Function f signature is: (fn [rec recs] ..) -> rec"
  [ks f tbl]
  (reduce
   (fn [res [vals subtbl]]
     (if-let [ok-rec (f (zipmap ks vals) subtbl)]
       (conj res ok-rec)
       res))
   []
   (index ks tbl)))
