;   Copyright (c) Walter van der Laan, 2010. All rights reserved.

;   The use and distribution terms for this software are covered by the
;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;   which can be found in the file epl-v10.html at the root of this 
;   distribution.
;   By using this software in any fashion, you are agreeing to be bound by
;   the terms of this license.
;   You must not remove this notice, or any other, from this software.

(ns net.wvdlaan.kiezel
  "Spreadsheet transformation library"
  (:use [clojure.contrib.command-line :only [with-command-line]]
	[net.wvdlaan.xls])
  (:require [net.wvdlaan.table :as t :exclude [unique]]))

;; logging

(def ^{:private true}
     lvl-order {:trace  1
		:debug  2
		:info   3
		:warn   4
		:error  5
		:fatal  6}) ; logging levels

(def ^{:private true}
     lvl-curr (ref (lvl-order :info))) ; current logging level

(def ^{:private true}
     msg-max (ref 0)) ; maximum number of logged messages

(def ^{:private true}
     msg-log (ref [])) ; message log

(defn- table?
  "Test if tbl is a collection of maps."
  [tbl]
  (and (coll? tbl) (map? (first tbl))))

(defn log
  "Return arg without making any changes to it.
  Add msg to the message log as a side effect, but, only if
  level lvl is equal to or greater than the level set with
  start-log."
  [lvl msg arg]
  (when (not (lvl-order lvl))
    (throw (Exception. (str "Unknown log level " lvl))))
  (dosync
   (when (or (= lvl :fatal)
	     (and
	      (<= @lvl-curr (lvl-order lvl))
	      (< (count @msg-log) @msg-max)))
     (let [t (table? arg)
	   r (if t (count arg) (if (map? arg) (get arg :$row nil) nil))
	   a (if t (first arg) arg)]
       (alter msg-log conj
	      {:level   lvl
	       :message msg
	       :row     r
	       :time    (java.util.Date.)
	       :detail  a}))))
  arg)

(def log-format [:level :message :row :time "mm:ss.00" :detail])

(defn get-log
  "Return the table containing all logged messages."
  [] @msg-log)

(defn start-log
  "Start, or restart, the message log.
  Log a maximum of max messages at level lvl or higher."
  ([lvl max]
     (start-log lvl max nil))
  ([lvl max arg]
     (dosync
      (ref-set lvl-curr (lvl-order lvl))
      (ref-set msg-max max)
      (ref-set msg-log []))
     arg))

(defn log-level
  "Change the minimum level set for logging messages."
  ([lvl]
     (log-level lvl nil))
  ([lvl arg]
     (dosync (ref-set lvl-curr (lvl-order lvl))) arg))

(defn dump
  "Dump one or more tables into 'dump.xls' and return
  the last argument."
  [& args]
  (poi-write (reduce
	      (fn [m [k t]]
		(assoc m k (sheet t)))
	      {}
	      (partition 2 args))
	     "dump.xls")
  (last args))

;; table with logging

(defn as       [& args] (log :trace "as"       (apply t/as args)))
(defn deselect [& args] (log :trace "deselect" (apply t/deselect args)))
(defn from     [& args] (log :trace "from"     (apply t/from args)))
(defn group    [& args] (log :trace "group"    (apply t/group args)))
(def  index    t/index)
(defn lookup   [& args] (log :trace "lookup"   (apply t/lookup args)))
(defn order-by [& args] (log :trace "order-by" (apply t/order-by args)))
(defn select   [& args] (log :trace "select"   (apply t/select args)))

(defn unique
  "Returns a function that logs situations were lookup does
  not find an exact match in the lookup-table."
  ([f]
     (unique "unique lookup" f))
  ([msg f]
     (unique :error :warn msg f))
  ([lvl-0 lvl-n msg f]
     (fn [rec1 recs2 k1 k2 v]
       (let [cnt (count recs2)
	     lvl (if (= cnt 0) lvl-0 lvl-n)]
	 (when (not= cnt 1) (log lvl msg rec1))
	 (if (= cnt 0) nil (f rec1 (first recs2)))))))

(defn enforce [& args]
  (let [tbl (last args)]
    (reduce
     (fn [res rec]
       (if-let
	   [checked
	    (loop [chk (first args) todo (rest args) m rec]
	      (if (< 0 (count todo))
		(let [k (chk 0) f (chk 1)]
		  (if-let [v (f (k m))]
		    (recur (first todo) (rest todo) (assoc m k v))
		    (if-let [lvl (get chk 2)]
		      (do (log lvl
			       (get chk 3 (str "enforce " (name k)))
			       rec)
			  nil)
		      nil)))
		m))]
	 (conj res checked)
	 res))
     []
     tbl)))

(defn to-double [this]
  (try
    (if (string? this)
      (Double/parseDouble this)
      (double this))
    (catch Exception _ nil)))

(defn to-date [this]
  (try
    (java-date this)
    (catch Exception _ nil)))

(defn to-string [^String this]
  (try
    (.toString this)
    (catch Exception _ nil)))

;; read & write sheet-list

(defn- get-path [^String dir ^String fname]
  (.getPath (java.io.File. dir fname)))

(defn- io-list [io fname tbl]
  (->> tbl
       (filter (fn [rec] (:name rec)))
       (filter (fn [rec] (= (:io rec) io)))
       (map (fn [rec]
	      (if (:sheet rec)
		rec
		(assoc rec :sheet (:name rec)))))
       (map (fn [rec]
	      (if (:book rec)
		rec
		(assoc rec :book fname))))
       (map (fn [rec]
	      (if (:cell rec)
		rec
		(assoc rec :cell "A1"))))))

(defn read-sheets
  "Returns a map of input tables as specified in sheet sheet-list
  of workbook dir/fname."
  [dir fname]
  (if-let [sdata (get (poi-read (get-path dir fname)
				#{"sheet-list"})
		      "sheet-list")]
    (let [sheet-list (table sdata)
	  idx (->> sheet-list
		   (io-list "in" fname)
		   (index [:book]))]
      (reduce
       (fn [res [[book] recs]]
	 (let [snames (set (map :sheet recs))
	       wbk (poi-read (get-path dir book) snames)]
	   (reduce
	    (fn [db rec]
	      (if-let [tname (rec :name)]
		(let [sname (rec :sheet)]
		  (if-let [sht (get wbk sname)]
		    (assoc db tname
			   (log :debug (str "read-sheets: " tname)
				(table sht (rec :cell))))
		    (do
		      (log :warn (str "read-sheets: " tname " not found")
			   (str "book= '" book "' sheet= '" sname "'"))
		      db)))
		db))
	    res
	    recs)))
       {"sheet-list" sheet-list}
       idx))
    (throw
     (Exception. (str "Cannot find sheet 'sheet-list' in book " fname)))))

(defn table-arg
  "Returns the value of column fld for table tbl as found in
  the table specification tbls."
  [tbl fld tbls]
  (log :debug (str "table-arg " fld)
       (fld (first (filter #(= tbl (% :name)) (tbls "sheet-list"))))))

(defn get-columns [todo]
  (vec (filter keyword? todo)))

(defn get-formats [todo]
  (loop [done []
	 f (first todo)
	 s (second todo)
	 todo (rest todo)]
    (if f
      (if (keyword? s)
	(recur (conj done nil)
	       s
	       (second todo)
	       (rest todo))
	(recur (conj done s)
	       (second todo)
	       (first (rest (rest todo)))
	       (drop 2 todo)))
      done)))

(defn- name2sheet [fs recs]
  (reduce
   (fn [res rec]
     (if-let [f (fs (:name rec))]
       (assoc res (:sheet rec) f)
       res))
   {}
   recs))

(defn- mapf [m f]
  (reduce (fn [res [k v]] (assoc res k (f v))) {} m))

(defn write-sheets
  "Writes tbls to workbook(s) as specified by sheet-list
  tbls       : database
  sheet-list : table-data"
  [dir fname frmts sheet-list tbls]
  (let [os (mapf frmts get-columns)
	fs (mapf frmts get-formats)]
    (->> sheet-list
	 (io-list "out" fname)
	 (group [:book]
		(fn [rec1 recs2]
		  (poi-write
		   (reduce
		    (fn [m rec2]
		      (if-let [tname (:name rec2)]
			(assoc m
			  (or (:sheet rec2) tname)
			  (sheet (t/deselect #{:$row} (tbls (:name rec2)))
				 (get os (:name rec2) [])))
			m))
		    {}
		    recs2)
		   (get-path dir (or (:book rec1) fname))
		   :sheet
		   (name2sheet fs recs2))))))
  nil)

;; main apps

(defn base-app [f frmts & args]
  (with-command-line args
    "Parameter: path/to/workbook.xls"
    [[level "Log level: trace, debug, warn, error or fatal" "error"]
     [max "Maximum number of log messages" 100]
     params]
    (let [^String wbk (params 0)
	  can (.getCanonicalFile (java.io.File. wbk))
	  dir (.getParent can)
	  fnm (.getName can)]
      (.delete (java.io.File. "dump.xls"))
      (start-log (keyword level) max)
      (log :debug "Base directory" dir)
      (log :debug "Base workbook" fnm)
      (let [dbi (read-sheets dir fnm)
	    dbo (f dbi)]
	(write-sheets dir fnm frmts
		      (dbi "sheet-list")
		      (assoc dbo "log" (get-log))))))
  nil)

(defn dev-app [f frmts & args]
  (try
    (apply base-app f frmts args)
    (catch Exception e
      (.printStackTrace e)
      (let [log (get-log)]
	(when (not= 0 (count log))
	  (poi-write
	   {"log" (sheet (get-log))} "dump.xls" :sheet
	   {"log" (get-formats log-format)}))))))

(defn main-app [f frmts & args]
  (try
    (apply base-app f frmts args)
    (catch Exception e
      (println "Fatal error " (.getMessage e)))))
