(ns net.molequedeideias.inga.ui
  (:require [net.molequedeideias.inga :as inga]
            [clojure.spec.alpha :as s]
            [net.molequedeideias.inga.task :as inga.task]
            [clojure.string :as string]))

(defn fqname
  [ident]
  (str (when (qualified-ident? ident)
         (str (namespace ident) "/"))
       (when (ident? ident)
         (name ident))))

(def form->params
  '{clojure.core/boolean? {:type "checkbox"}
    clojure.core/nat-int? {:type "number"
                           :min  "0"}
    clojure.core/pos-int? {:type "number"
                           :min  "1"}})

(defn spec->input-params
  [x]
  (try
    (form->params (s/form x) {})
    (catch Exception ex
      (if (string/starts-with? (ex-message ex)
                               "Unable to resolve spec")
        {}
        (throw ex)))))

(defn ui-input-keyword
  ([kv] (ui-input-keyword kv false))
  ([[k v vs key-labels] hidden?]
   (let [list-id (str (gensym))]
     [:label.form-group.mb-3
      (when hidden?
        {:hidden true})
      (get key-labels k
           (pr-str k))
      (if (coll? vs)
        [:select {:name (fqname k)}
         (for [i vs]
           [:option {:value (str i)}
            (get key-labels i
                 (pr-str i))])]
        [:input.form-control
         (cond-> (assoc (spec->input-params k)
                   :name (fqname k))
                 (boolean v) (assoc :value v)
                 (true? v) (assoc :checked v)
                 (coll? vs) (assoc :list list-id)
                 :always (assoc :id (pr-str k)))])])))

(defn ui-form
  [{::keys [prefix mutation key-labels ks anti-forgery-token kvs submit-label hidden? show select-values disabled? hidden-field?]
    :or    {hidden-field? (constantly false)}}]
  (let [ks (distinct (concat (if (map? show)
                               (vals show)
                               show)
                             (or ks (keys kvs))))
        select-values (if (empty? select-values)
                        (select-keys kvs (if (map? show)
                                           (vals show)
                                           show))
                        select-values)]
    [:form
     (cond-> {:style {:padding "1rem"}}
             mutation (assoc :action (str
                                       (when prefix
                                         (str "/" prefix))
                                       "/" mutation)
                             :method :post)
             (not mutation) (assoc :method :get))
     (for [k ks
           :let [hidden? (if hidden?
                           (not (contains? show k))
                           false)]]
       (ui-input-keyword [k
                          (get kvs k)
                          (when-not hidden?
                            (get select-values (get show k k)))
                          key-labels]
                         (or hidden?
                             (hidden-field? k))))
     (when (and mutation anti-forgery-token)
       [:input {:type :hidden :name "__anti-forgery-token" :value anti-forgery-token}])
     [:input.form-control
      (cond-> {:class "btn btn-primary"
               :value (str (or submit-label
                               (get key-labels mutation)
                               mutation ">"))
               :type  :submit}
              disabled? (assoc :disabled true))]]))

(defn ui-kv-table
  [{::keys [query data ask-for anti-forgery-token prefix key-labels]}]
  [:tbody
   (for [qel query]
     [:tr
      [:th
       (get key-labels qel (pr-str qel))]
      [:td
       (let [v (get data qel)]
         (cond
           (symbol? qel) (ui-form {::mutation           qel
                                   ::hidden?            true
                                   ::prefix             prefix
                                   ::show               (get ask-for qel)
                                   ::anti-forgery-token anti-forgery-token
                                   ::kvs                data})
           (string? v) v
           (contains? key-labels v) (get key-labels v)
           (number? v) v
           :else (pr-str v)))]])])

(defn ui-vs-table
  [{::keys [query data ident->href-fn anti-forgery-token ask-for prefix key-labels]}]
  (list
    [:thead
     [:tr
      (for [el query]
        [:th
         (get key-labels el (pr-str el))])]]
    [:tbody
     (for [el data]
       [:tr
        (for [qel query]
          [:td
           (let [v (get el qel)
                 href-fn (get ident->href-fn qel)]
             (cond
               (symbol? qel) (ui-form {::mutation           qel
                                       ::hidden?            true
                                       ::key-labels         key-labels
                                       ::prefix             prefix
                                       ::show               (get ask-for qel)
                                       ::anti-forgery-token anti-forgery-token
                                       ::kvs                el})
               (fn? href-fn) [:a {:href (href-fn v)} (str v)]
               (boolean? v) [:input {:type     "checkbox"
                                     :checked  v
                                     :disabled true}]
               (contains? key-labels v) (get key-labels v)
               (string? v) v
               (number? v) v
               :else [:code (pr-str v)]))])])]))


(defn ui-table-pagination
  [{:edn-query-language.pagination/keys [first-element-index
                                         last-element-index
                                         total-count]}]
  [:div
   (list
     [:span (str first-element-index)]
     [:span "-"]
     [:span (str last-element-index)]
     [:span ":"]
     [:span (str total-count)])])

(defn ui-queried-render
  [{::keys [query data single? paginated? ident->href-fn key-labels anti-forgery-token ask-for prefix]}]
  (list
    [:div
     {:class "container-fluid table-responsive p-0"}
     [:table {:class "table"}
      (cond
        single? (ui-kv-table {::query              query
                              ::ask-for            ask-for
                              ::key-labels         key-labels
                              ::prefix             prefix
                              ::anti-forgery-token anti-forgery-token
                              ::data               (first (:edn-query-language.pagination/edges data))})
        :else (ui-vs-table {::query              query
                            ::ident->href-fn     ident->href-fn
                            ::ask-for            ask-for
                            ::prefix             prefix
                            ::key-labels         key-labels
                            ::anti-forgery-token anti-forgery-token
                            ::data               (:edn-query-language.pagination/edges data)}))]]))

(defn ui-btn-link
  [{:keys [href label]}]
  [:a
   {:class "btn btn-outline-secondary "
    :href  href
    :style {:margin-left "1rem"}}
   label])

(defn ui-downloads
  [downloads]
  [:div
   {:class "container-fluid d-flex flex-row-reverse bd-highlight"
    :style {:padding-bottom "1rem"}}
   (map ui-btn-link downloads)])

(defmethod inga/render ::query
  [env]
  (assoc env
    ::inga/query [::inga.task/query-params
                  ::inga.task/anti-forgery-token
                  ::inga.task/display-properties
                  ::inga.task/data
                  ::inga.task/single?
                  ::inga.task/paginated?
                  ::inga.task/ident->href-fn
                  ::inga.task/downloads
                  ::inga.task/default-params
                  ::inga.task/key-labels
                  ::inga.task/ask-for
                  ::inga.task/prefix
                  ::inga.task/params]
    ::inga/ui (fn [{::inga.task/keys [anti-forgery-token query-params key-labels
                                      display-properties data single? paginated?
                                      ident->href-fn downloads default-params ask-for prefix params]}]
                (let [hidden-field? (if paginated?
                                      (comp #{"edn-query-language.pagination"} namespace)
                                      (constantly false))]
                  (list
                    (when-not (empty? (remove hidden-field? params))
                      (ui-form {::ks            params
                                ::hidden-field? hidden-field?
                                ::prefix        prefix
                                ::key-labels    key-labels
                                ::kvs           (merge default-params query-params data)}))
                    (ui-downloads downloads)
                    (ui-queried-render {::query              display-properties
                                        ::key-labels         key-labels
                                        ::single?            single?
                                        ::ident->href-fn     ident->href-fn
                                        ::ask-for            ask-for
                                        ::prefix             prefix
                                        ::anti-forgery-token anti-forgery-token
                                        ::paginated?         paginated?
                                        ::data               data})
                    (when paginated?
                      (let [{:edn-query-language.pagination/keys [current-page]
                             :as                                 kvs} (merge default-params query-params data)]
                        [:div
                         {:style {:display "flex"}}
                         (ui-table-pagination data)
                         (ui-form {::ks           params
                                   ::prefix       prefix
                                   ::key-labels   key-labels
                                   ::submit-label "<-"
                                   ::hidden?      true
                                   ::disabled?    (not (pos? current-page))
                                   ::kvs          (assoc kvs :edn-query-language.pagination/current-page (dec current-page))})
                         (ui-form {::ks           params
                                   ::prefix       prefix
                                   ::key-labels   key-labels
                                   ::submit-label "->"
                                   ::hidden?      true
                                   ::kvs          (assoc kvs :edn-query-language.pagination/current-page (inc current-page))})])))))))


(defn edit->query
  [{::inga/keys [edit-with display-properties]
    :as         env}]
  (let [ask-for (into {} (for [[prop mutation] edit-with]
                           [mutation #{prop}]))]
    (assoc env
      ::inga/display-properties (into display-properties
                                      (vals edit-with))
      ::inga/ask-for ask-for
      ::inga/render ::query)))

(defmethod inga/render ::edit
  [env]
  (inga/render (edit->query env)))

(defmethod inga/render ::mutation
  [env]
  (assoc env
    ::inga/query [::inga.task/query-params
                  ::inga.task/anti-forgery-token
                  ::inga.task/prefix
                  ::inga.task/select-values
                  ::inga.task/args
                  ::inga.task/mutation]
    ::inga/ui (fn [{::inga.task/keys [query-params mutation select-values prefix anti-forgery-token args]}]
                (ui-form {::ks                 args
                          ::kvs                query-params
                          ::select-values      select-values
                          ::prefix             prefix
                          ::anti-forgery-token anti-forgery-token
                          ::mutation           mutation}))))


(defn register
  []
  (concat (inga.task/register)
          []))
