(ns net.molequedeideias.inga
  (:require [hiccup2.core :as h2]
            [com.wsscode.pathom.core :as p]
            [com.wsscode.pathom.connect :as pc]
            [ring.util.mime-type :as mime]
            [io.pedestal.http :as http]
            [net.molequedeideias.inga.rest :as rest]
            [reitit.core :as r]
            [spec-coerce.core :as sc]))

(defn parse-boolean
  [x]
  (case x
    "true" true
    "false" false
    "on" true
    x))


(defmethod sc/sym->coercer `boolean? [_] parse-boolean)

(defn match-by-path
  [{::keys [reitit by-name]} path]
  (let [match (r/match-by-path reitit path)
        name* (-> match :data :name)]
    (into (by-name name*)
          (map (fn [[k v]]
                 [(keyword
                    "net.molequedeideias.inga"
                    (name k))
                  v]))
          match)))

(defn match-by-name
  [{::keys [reitit by-name]} name* path-params]
  (let [match (r/match-by-name reitit name* path-params)
        name* (-> match :data :name)]
    (into (by-name name*)
          (map (fn [[k v]]
                 [(keyword
                    "net.molequedeideias.inga"
                    (name k))
                  v]))
          match)))

(defn ->router
  [routes]
  (let [routes (for [[k v] routes]
                 (assoc v ::name k))]
    {::reitit  (r/router (into []
                               (map (juxt ::path ::name))
                               routes))
     ::by-name (into {}
                     (map (juxt ::name identity)
                          routes))}))


(defmulti render ::render)

(defn std-parser
  [{::keys [register]
    :as    args}]
  (p/parser
    {::p/mutate  pc/mutate
     ::p/env     (assoc args ::p/reader [p/map-reader
                                         pc/reader2
                                         pc/open-ident-reader
                                         p/env-placeholder-reader]
                             ::p/placeholder-prefixes #{">"})
     ::p/plugins [(pc/connect-plugin {::pc/register (if (fn? register)
                                                      (register)
                                                      register)})
                  p/elide-special-outputs-plugin
                  p/error-handler-plugin]}))

(defn +parser
  [{::keys [parser]
    :as    args}]
  {:name  ::+parser
   :enter (fn [ctx]
            (let [parser (or parser
                             (std-parser args))]

              (-> ctx
                  (assoc-in [:request ::parser] parser)
                  (update :request (fn [req]
                                     (merge args req))))))})

(defn index-get
  [{:keys  [path-info]
    ::keys [parser]
    :as    req}]
  (let [{::keys [title
                 theme-color
                 icon
                 lang
                 description
                 router]} (parser req
                                  [::title
                                   ::theme-color
                                   ::lang
                                   ::description
                                   ::icon
                                   ::router])
        {::keys [contents]
         :as    router-data} (when router
                               (match-by-path (->router router)
                                              path-info))]
    (when contents
      {:body    (str (h2/html
                       (when lang
                         {:lang lang})
                       [:html
                        [:head
                         [:meta {:name    "viewport"
                                 :content "width=device-width, initial-scale=1.0"}]
                         (when description
                           [:meta {:name "Description" :content description}])
                         (when theme-color
                           [:meta {:name "theme-color" :content theme-color}])
                         (when icon
                           (list
                             [:link {:rel  "apple-touch-icon"
                                     :href icon}]
                             [:link {:rel  "shortcut icon"
                                     :href icon}]))
                         (when title
                           [:title title])
                         [:meta {:charset "utf-8"}]
                         [:link {:rel  "stylesheet"
                                 :href "/bootstrap/4.4.1/dist/css/bootstrap.min.css"}]]
                        [:body.container-fluid
                         {:style {:background-color "#f8f9fa"}}
                         (for [content contents
                               :let [{::keys [query ui]
                                      :as    env} (render (merge req content router-data {::parser parser}))]]
                           [:div
                            {:style {:margin  "1rem"
                                     :padding "1rem"
                                     :border  "1px solid"}}
                            (ui (parser env query))])
                         [:script {:src "/jquery/3.4.1/dist/jquery.slim.min.js"}]
                         [:script {:src "/popper.js/1.16.0/dist/umd/popper.min.js"}]
                         [:script {:src "/bootstrap/4.4.1/dist/js/bootstrap.min.js"}]]]))

       :headers {"Cache-Control"           "no-store"
                 "Content-Security-Policy" ""
                 "Content-Type"            (mime/default-mime-types "html")}
       :status  200})))

(defn drop-prefix
  [prefix]
  {:name  ::drop-prefix
   :enter (fn [ctx]
            (update-in ctx [:request :path-info] (fn [path-info]
                                                   (if prefix
                                                     (subs path-info (inc (count prefix)))
                                                     path-info))))})

(defn export
  ([args] (export nil args))
  ([prefix {::keys [register] :as args}]
   (let [parser (when-not (fn? register)
                  (std-parser args))
         args (assoc args
                ::parser parser)]
     `#{[~(str "/" prefix) :get [~(drop-prefix prefix) ~(+parser args) index-get]
         :route-name ~(keyword "net.molequedeideias.inga" (str "index-" prefix))]
        [~(str (when prefix
                 (str "/" prefix))
               "/*path") :get [~(drop-prefix prefix) ~(+parser args) index-get rest/get-rest]
         :route-name ~(keyword "net.molequedeideias.inga" (str "index*-" prefix))]
        [~(str (when prefix
                 (str "/" prefix))
               "/*path") :post [~(drop-prefix prefix) ~(+parser args) rest/index-post]
         :route-name ~(keyword "net.molequedeideias.inga" (str "post-" prefix))]})))

(defn generate-routes
  [{::http/keys [routes]
    ::keys      [apps]
    :or         {routes #{}}
    :as         service-map}]
  (let [routes (into routes
                     cat
                     (for [{::keys [prefix] :as env} apps]
                       (export prefix (merge service-map
                                             env))))]
    (assoc service-map ::http/routes routes)))
