(ns
  ^{:author "mikera"
    :doc "Special effects. Images can be generated by composing these functions."}  
  clisk.effects
  (:use [clisk node util functions colours patterns textures]))

(set! *warn-on-reflection* true)
(set! *unchecked-math* :warn-on-boxed)

(defn psychedelic 
  "Psychedelic colour effect"
  ([src & {:keys [noise-scale noise-bands] 
           :or {noise-scale 0.2 noise-bands 2.0}}]
    (adjust-hue (v* noise-bands (scale noise-scale noise)) src)))

(defn monochrome
  "Converts to monochrome"
  ([src]
    (lightness-from-rgb src)))

(defn posterize
  "Posterization effect with discrete colour bands."
  ([src & {:keys [bands] 
           :or {bands 4}}]
    (let [bands (double bands)
          inv-bands (/ 1.0 (dec bands))
          dec-bands (- bands 0.00001)] 
      (warp
        src 
        (v* inv-bands (vfloor (v* dec-bands pos)))))))

(defn pixelize
  "Pixelizes an image with pixels of given size"
  [size src]
  (scale size (warp vfloor (scale (/ 1.0 (double size)) src))))

(defn shatter 
  "Breaks an image up into sharp irregular fragments defined by a Voronoi map" 
  [src & args]
  (offset (grain (apply voronoi-points args)) src))

(defn radial 
  "Wraps an image around an origin in a radial fashion in the x-y plane"
  [src & {:keys [repeat] 
          :or {repeat 1.0}}]
  (warp [(vfrac (v* repeat (vdivide theta TAU))) , radius] 
        src))

(defn swirl
  "Swirls a function around the origin with a given rate"
  ([src]
    (swirl 1.0 src))
  ([rate src]
    (rotate (v* rate (vsqrt length) ) src)))

(defn add-glow 
  "Adds a glow effect to any image with an alpha channel"
  [image-with-alpha 
   & {:keys [glow-radius glow-colour glow-background blur-image-size] 
      :or {glow-radius 0.2 
           glow-colour yellow
           glow-background [0.0 0.0 0.0 0.0]
           blur-image-size 256}}]
  (let [glow-radius (double glow-radius)
        blur-image-size (long blur-image-size)
        alpha-channel (alpha image-with-alpha)
        blur-filter (doto (com.jhlabs.image.GaussianFilter.) 
                          (.setRadius (int (* glow-radius blur-image-size))))
        blur (image-filter blur-filter alpha-channel :size blur-image-size)
        glow-effect (lerp blur glow-background glow-colour)]
    (lerp alpha-channel glow-effect image-with-alpha)))