(ns talk.core
  (:refer-clojure :exclude [+ - * / zero? partial ref])
  (:require [sicmutils.env :refer :all]
            [sicmutils.infix :as infix]
            [sicmutils.examples.driven-pendulum :as driven]
            [sicmutils.examples.double-pendulum :as double]
            [hiccup.core :refer :all]
            [hiccup.page :refer [html5]])
  (:gen-class))

(defmacro eval-track
  "Provided a sequence of forms, evaluates to a sequence of pairs [form, value]."
  [& forms]
  `(map vector (quote ~forms) (vector ~@forms)))

(defn -main
  "Generate the presentation HTML."
  [& args]

  (let [exprs (eval-track

               (+ 'x 'y)

               (+ -1 ((expt sin 2) 'psidot))
               (defn w-line [t]
                 (up (- 1 t)
                     t))

               (defn wdot-line [t]
                 (up -1 1))

               (defn w-arc [t]
                 (up (cos (* 1/2 pi t))
                     (sin (* 1/2 pi t))))

               (defn wdot-arc [t]
                 (up (* -1/2 pi (sin (* 1/2 pi t)))
                     (* 1/2 pi (cos (* 1/2 pi t)))))

               (defn L-free [t q qdot]
                 (* 1/2 (square (qdot t))))

               (defn L-central [t q qdot]
                 (- (L-free t q qdot) (/ (sqrt (square (q t))))))

               (defn action [L w wdot]
                 (definite-integral #(L % w wdot) 0 1))

               (w-line 0)
               (w-line 1)
               (w-arc 0)
               (w-arc 1)
               (action L-free w-line wdot-line)
               (action L-free w-arc wdot-arc)


               (action L-free w-line (D w-line))
               (action L-free w-arc (D w-arc))


               (action L-central w-line (D w-line))
               (action L-central w-arc (D w-arc))

               (w-line 't)
               ((D w-line) 't)

               (defn L-free [[_ q qdot]]
                 (* 1/2 (square qdot)))

               (def w
                 (up (literal-function 'x)
                     (literal-function 'y)
                     (literal-function 'z)))
               (w 't)
               (L-free ((Gamma w) 't))

               (((Lagrange-equations L-free) w) 't)

               (((Lagrange-equations L-free) w) 't)

               (down 0 0 0)

               [(down 0 0 0)]

               (with-literal-functions [θ]
                 (((Lagrange-equations
                    (driven/L 'm 'l 'g 'a 'ω))
                   θ)
                  't))

               ((Lagrangian->state-derivative
                 (driven/L 'm 'l 'g 'a 'ω))
                (up 't 'θ 'θdot))

               [(with-literal-functions [θ φ]
                  (((Lagrange-equations ((- double/T double/V) 1 1 1 1 'g))
                    (up θ φ))
                   't))]

               [((Lagrangian->state-derivative
                  ((- double/T double/V) 1 1 1 1 'g))
                 (up 't (up 'θ 'φ) (up 'θdot 'φdot)))]

               (defn vertical-periodic-drive
                 [a ω]
                 #(* a (cos (* ω %))))

               (defn T-pend [m l g ys]
                 (let [vys (D ys)]
                   (fn [[t θ θdot]]
                     (let [vyst (vys t)]
                       (* 1/2 m
                          (+ (square (* l θdot))
                             (square vyst)
                             (* 2 l vyst θdot (sin θ))))))))

               (defn V-pend [m l g ys]
                 (fn [[t θ θdot]]
                   (* m g (- (ys t) (* l (cos θ))))))

               (def L-pend (- T-pend V-pend))

               (defn L-periodically-driven-pendulum [m l g a ω]
                 (L-pend m l g (vertical-periodic-drive a ω)))

               ((Lagrangian->state-derivative
                 (L-periodically-driven-pendulum 'm 'l 'g 'a 'ω))
                (up 't 'θ 'θdot))

               ((Hamiltonian->state-derivative
                 (Lagrangian->Hamiltonian
                  (L-periodically-driven-pendulum 'm 'l 'g 'a 'ω)))
                (up 't 'θ 'p_θ))

               (up 't (up 'θ 'φ) (up 'θdot 'φdot))
               )]

    (spit
     "target/prez2.html"
     (html5
      [:head
       [:script {:src "https://cdnjs.cloudflare.com/ajax/libs/KaTeX/0.7.1/katex.min.js"}]
       [:style "pre.code { font-family: 'Inconsolata', monospace; }"]
       [:link {:rel "stylesheet" :href "https://fonts.googleapis.com/css?family=Inconsolata"}]
       [:link {:rel "stylesheet" :href "https://cdnjs.cloudflare.com/ajax/libs/KaTeX/0.7.1/katex.min.css"}]]
      [:body
       (for [[form value] exprs]
         [:div.eval-pair
          [:pre.code (str form)]
          [:katex (if (vector? form)
                    (binding [infix/*TeX-vertical-down-tuples* true]
                      (-> value first simplify ->TeX str))
                    (-> value simplify ->TeX str))]])
       [:script {:type "text/javascript"}
        (str  "Array.prototype.forEach.call(document.getElementsByTagName(\"katex-display\"), "
              "function(e) {katex.render(e.textContent, e, {displayMode: true});});"
              "Array.prototype.forEach.call(document.getElementsByTagName(\"katex\"), "
              "function(e) {katex.render(e.textContent, e);});")]]))))


(-main)
