(ns sicmutils.calculus.manifold
  (:require [clojure.string :as str]
            [sicmutils
             [value :as v]]))

;; What is a manifold? It has a name and a base type, assumed to be 'Real.
;;

;; (defn coordinate-system
;;   [coordinate-system-name patch]
;;   ((patch 'get-coordinate-system) coordinate-system-name))

;; (defn coordinate-system-at
;;   [coordinate-system-name patch-name manifold]
;;   (coordinate-system coordinate-system-name (patch patch-name manifold)))


;; the object produced by this function answers the messages:


(defn make-patch
  [name generator setup]
  {:name name
   :generator generator
   :setup setup})

(defprotocol IManifoldSpecification
  (new-patch [this name generator setup])
  (patch-setup [this name])
  (generator [this]))

(defn specify-manifold
  [manifold-name & {:keys [over] :or {over 'Real}}]
  (let [counter (atom 0)
        name (symbol (str (let [namestring (str manifold-name)]
                            (if (re-matches #".*\^n" namestring)
                              (str/replace namestring #"n$" (str dimension)))
                            namestring)
                          "-"
                          counter))]
    {:type      ::manifold-specification
     :over      over
     :patch     {}
     :generator (fn [dimension & {:keys [embedding-dimension]}]
                  {:dimension           dimension
                   :embedding-dimension (or embedding-dimension dimension)
                   :name                name})
     }))


;; specify-manifold produces an object answering the SETUP api.
;;   new-patch(new-patch-name, patch-generator, patch-setup)
;;     -- mutates manifold by adding a patch
;;        this just means we add the argument triple to the patches list.
;;   patch-setup(patch-name) -- returns the setup entry of the named patch.
;;   generator() -- returns generator object.

;; generator takes (dimension & optionally embedding-dimension) to produce
;; a manifold object with that data baked in.

;; the manifold object answers the following API:
;;   name
;;   manifold (i.e., self?)
;;   type (local)
;;   dimension or manifold-dimension --> dimension (local)
;;   embedding-dimension --> ditto
;;   get-patch
;;   distinguished points
;;   add-distinguished-point! --> (does anyone call this? apparently not)
;;   patch-names (from patches list)



