(ns views.honeysql.core
  (:import
    (clojure.lang Atom))
  (:require
    [views.core :refer [hint put-hints!]]
    [views.honeysql.util :refer [query-tables]]
    [honey.sql :as hsql]
    [clojure.tools.logging :refer [error]]
    [clojure.java.jdbc :as j]))

(def hint-type :sql-table-name)

(defmacro with-view-transaction
  "Works exactly like clojure.java.jdbc/with-db-transaction. Use this instead to wrap
   vexec! calls which need to be run in a transaction. Holds all view system hints
   generated by any vexec! calls within the transaction until the end, at which point
   they are all sent to the view system."
  [^Atom view-system binding & forms]
  (let [tvar (first binding)
        db   (second binding)
        args (drop 2 binding)]
    `(if (:views-honeysql/hints ~db) ;; check if we are in a nested transaction
       (let [~tvar ~db] ~@forms)
       (let [hints#   (atom [])
             result#  (j/with-db-transaction [t# ~db ~@args]
                                             (let [~tvar (assoc ~db :views-honeysql/hints hints#)]
                                               ~@forms))]
         (put-hints! view-system @hints#)
         result#))))

(defn- execute-honeysql!
  "Always return keys for inserts."
  [db hsql-map]
  (if-let [table (:insert-into hsql-map)]
    (if (vector? table)
      (j/execute! db (hsql/format hsql-map))
      (apply j/insert! db table (:values hsql-map)))
    (j/execute! db (hsql/format hsql-map))))

(defn vexec!
  "Used to run any SQL insert/update/delete query on the database while ensuring
   that view hints are sent to the view system to trigger any relevant view
   refreshes. Use this instead of calling clojure.java.jdbc/execute! or
   clojure.java.jdbc/insert!. If you need to perform an operation in a transaction
   use with-view-transaction.

   Arguments are:
   - db: a clojure.java.jdbc database connection
   - action-map: the HoneySQL map for the insert/update/delete SQL query

   Options are:
   - namespace: a namespace that will be included in the hints sent out"
  [^Atom view-system db action-map & [{:keys [namespace] :as options}]]
  (let [results   (execute-honeysql! db action-map)
        hsql-hint (hint namespace (query-tables action-map) hint-type)]
    (if-let [hints (:views-honeysql/hints db)]
      (swap! hints conj hsql-hint)
      (put-hints! view-system [hsql-hint]))
    results))
