(ns net.ftod.zcube
  ( :import net.ftod.zcube.zdd.ZDD
            net.ftod.zcube.zdd.ZDDNumber
            net.ftod.zcube.zdd.ZDDTree
            net.ftod.zcube.zdd.ZDDTerm
            java.lang.Iterable
            java.util.Collection
  )
)

;
; The algebra of ZDD trees.
;

( def ^ZDDTree top ZDDTree/TOP ) ; The singleton set containing the empty tree. 
( def ^ZDDTree bot ZDDTree/BOT ) ; The empty set of trees.

( defn ^ZDDTree prefix [ ^Iterable strings ^ZDDTree tree ]
  "Prefix a set of trees with a segment."
  ( ZDDTree/prefix strings tree )
)

( defn ^ZDDTree path [ & strings ]
  "Build a singleton containing a linear tree."
  ( ZDDTree/path ^Iterable strings )
)

( defn ^ZDDTree cross [ & trees ]
  "Generate new trees as the cross union of the trees in the treesets."
  ( ZDDTree/cross ^Collection trees )
)

( defn ^ZDDTree sum [ & trees ]
  "Take union of tree sets."
  ( ZDDTree/sum ^Collection trees )
) 

;
; The associative/commutative API
;

( defn ^ZDDNumber subtrees
  "Construct ZDD number counting the occurrences of subtrees of a tree.
  "
  [ ^ZDDTerm zt ] ( ZDDTerm/subtrees zt )
)

( defn ^ZDDNumber filter-subtrees
  "Construct ZDD number counting the occurrences of subtrees of a tree.
   This higher-order one-argument version takes a filter expressed a sequence of trees,
   and yields the corresponding constructor function.
  "
  [ trees ]
  ( let [ ^ZDD z ( ZDDTree/unionTrees trees ) ] ; Pay once the computation of the ZDD...
    ( fn [ ^ZDDTerm zt ] ( ZDDTree/subtrees z zt ) ) ; ...reuse it multiple times.
  )
)

( defn ^ZDDNumber add
  "Add two signed ZDD numbers. Associative and commutative, nil is the zero element."
  [ ^ZDDNumber zn1 ^ZDDNumber zn2 ]
  ( ZDDNumber/negabinaryAdd zn1 zn2 )
)

( defn ^ZDDNumber sub
  "Subtract two signed ZDD numbers."
  [ ^ZDDNumber zn1 ^ZDDNumber zn2 ]
  ( ZDDNumber/negabinarySub zn1 zn2 )
)

;
; The accumulative API
;

( defn ^ZDDNumber add-subtrees
  "Add occurrences of subtrees to a ZDD number."
  [ ^ZDDTerm zt ^ZDDNumber zn ] ( ZDDNumber/addSubtrees zt zn )
)

( defn ^ZDDNumber add-filter-subtrees
  "Add occurrences of subtrees to a ZDD number.
   This higher-order one-argument version takes a filter expressed a sequence of trees,
   and yields the corresponding adder function.
  "
  [ trees ]
  ( let [ ^ZDD z ( ZDDTree/unionTrees trees ) ] ; Pay the ZDD computation once...
    ( fn [ ^ZDDTerm zt ^ZDDNumber zn ] ( ZDDNumber/addSubtrees z zt zn ) ) ; ...possibly apply multiple times over ZDD numbers.
  )
)

;
; Retrieving counts from ZDD numbers.
;

( defn ^long count-trees
  "Count occurrences of trees in a ZDD number.
   This is an higher-order function that yields a proper counting function.
  "
  [ ^ZDDTree tree ]
  ( let [ ^ZDD z ( ZDDTree/trees tree ) ] ; Pay the ZDD computation once...
    ( fn [ ^ZDDNumber n ] ( ZDDNumber/negabinary n z ) ) ; ...possibly apply multiple times over ZDD numbers.
  )
)

;
; Bulk operations on sequences of ZDDTerm
;

( defn ^ZDDTerm times
  "Multiply a treeset by a (long) number of occurrences."
  [ ^long l ^ZDDTree t ]
  ( ZDDTerm/times l t )
)

( defn ^ZDDNumber sum-subtrees
  "Sum the occurrences of subtrees generated by a sequence of ZDDTerms."
  [ zts ]
  ( ZDDNumber/sumSubtrees ^Iterable zts )
)

( defn ^ZDDNumber p-sum-subtrees
  "Sum in parallel the occurrences of subtrees generated by a sequence of ZDDTerms."
  [ zts ]
  ( ZDDNumber/pSumSubtrees ^Iterable zts )
)

( defn sum-group-by
  "Sum a sequence of ZDDTerm, simultaneously grouping by a sequence of trees."
  [ ts zts ]
  ( seq ( ZDDNumber/sumGroupBy ( into-array ZDDTree ts ) ^Iterable zts ) )
)

( defn p-sum-group-by
 "Sum in parallel a sequence of ZDDTerm, simultaneously grouping by a sequence of trees."
 [ ts zts ]
 ( seq ( ZDDNumber/pSumGroupBy ( into-array ZDDTree ts ) ^Iterable zts ) )
)
