					; (c) 2010 David Soria Parra
					; Licensed under the terms of the MIT License

(ns
  #^{:doc "A simple CouchDB Wrapper",
     :author "David Soria Parra"}
  experimentalworks.couchdb
  (:use clojure.contrib.duck-streams,
	clojure.contrib.json.read,
	clojure.contrib.json.write)
  (:import java.net.URL))

(defn- build-result
  [conn]
  (let [result (slurp* (.getInputStream conn))]
    (merge
     (read-json result)
     {"code" (.getResponseCode conn)})))

(defn successful?
  "Tests if the result of a CouchDB call indicates that it was successful."
  [result]
  (let [code (result "code")]
    (or
     (= code 201)
     (= code 200))))

(defmacro with-conn
  "Assumes that the first binding is a connection and closes
  the connection after executing the body."
  [bindings & body]
  `(let ~bindings
     (let [res# (do ~@body)]
	(.disconnect ~(first bindings))
	res#)))

(defn couch-connection
  "Creates a new connection to a CouchDB server. The used method can
   be either :post, :get or :put."
  [url method]
  (let [conn (. (new URL url) openConnection)
	mapping {:post "POST" :get "GET" :put "PUT"}]
    (doto conn
      (.setRequestProperty "Content-Type" "application/json")
      (.setRequestMethod (mapping method))
      (.setDoOutput true)
      (.setDoInput true)
      (.connect))
    conn))

(defn couch-get
  "Get a document with the given id from the DB."
  [url id]
  (with-conn [conn (couch-connection (str url "/" id) :get)]
    (build-result conn)))

(defn couch-update
  "Update the given document id with the new struct."
  [url id struct]
  (let [data (json-str struct)]
    (with-conn [conn (couch-connection (str url "/" id) :put)]
      (spit (.getOutputStream conn) data)
      (build-result conn))))

(defn couch-add
  "Add a given struct to the CouchDB at the given url."
  [url struct]
  (let [data (json-str struct)]
    (with-conn [conn (couch-connection url :post)]
      (spit (.getOutputStream conn) data)
      (build-result conn))))
