(ns net.dnolen.vecmath.perf
  (:use net.dnolen.vecmath.core)
  (:require net.dnolen.vecmath.Vector2d
            net.dnolen.vecmath.Vector3d)
  (:import [net.dnolen.vecmath.Vector2d Vector2d]
           [net.dnolen.vecmath.Vector3d Vector3d]))

;; -----------------------------------------------------------------------------
;; Basic Stuff

(def v1 (Vector2d. 5.0 3.33))
; type hinting the following only makes it slower
(dotimes [_ 10]
  (time
   (dotimes [_ 1000000]
     (div v1 2.0))))

;; -----------------------------------------------------------------------------
;; Array of javax.vecmath.Vector2d vs Vector of vec2

(comment
  (defn #^"[Ljavax.vecmath.Vector2d;" make-v2d-array [n]
    (let [#^"[Ljavax.vecmath.Vector2d;" ary (make-array Vector2d n)]
      (amap ary i ary
            (aset ary i (Vector2d. 5.0 5.0)))))

  ;; ~50ms
  (dotimes [_ 10]
    (time (make-v2d-array 250000)))

  ;; ~40ms
  (dotimes [_ 10]
    (let [ary (make-v2d-array 250000)
          ov   (Vector2d. 2.5 2.5)]
      (time
       (amap ary i ary
             (let [#^Vector2d v (aget ary i)]
               (.add v ov)
               (aset ary i v))))))

  ;; ~50ms
  (dotimes [_ 10]
    (let [ary (make-v2d-array 250000)
          ov   (Vector2d. 2.5 2.5)]
      (time
       (amap ary i ary
             (let [#^Vector2d v (.clone #^Vector2d (aget ary i))]
               (.add v ov)
               (aset ary i v))))))
  )

(defn make-vec2-vector [n]
  (let [n (int n)]
   (loop [i (int 0) result (transient [])]
     (if (= i n)
       (persistent! result)
       (recur (inc i) (conj! result (Vector2d. 5.0 5.0)))))))

(comment
  ;; ~70ms, great
  (dotimes [_ 50]
    (time (make-vec2-vector 250000)))

  (dotimes [_ 10]
    (time (make-vec2-vector (* 150 150))))

  ;; ~110ms, not bad considering
  ;; we talking about immutability here
  (let [xs (make-vec2-vector 250000)
        ov (Vector2d. 2.5 2.5)]
   (dotimes [_ 50]
     (time
      (doall (map #(add % ov) xs)))))

  (def v1 (Vector3d. 5 9 6))
  (def v2 (Vector3d. 7 8 9))

  (dotimes [_ 10]
    (time
     (dotimes [_ 1000000]
       (unit v1))))
  
  (dotimes [_ 10]
    (time
     (dotimes [_ 1000000]
       (dot v1 v2))))

  (dotimes [_ 10]
    (time
     (dotimes [_ 1000000]
       (cross v1 v2))))
  )