(ns net.dnolen.vecmath.Vector3d
  (:use net.dnolen.vecmath.core
        net.dnolen.vecmath.utils))

;; -----------------------------------------------------------------------------
;; Vector3d

(defrecord Vector3d
  [#^double x #^double y #^double z]

  VecMath
  (add [_ other] (Vector3d. (+ x (.x #^Vector3d other))
                        (+ y (.y #^Vector3d other))
                        (+ z (.z #^Vector3d other))))

  (sub [_ other] (Vector3d. (- x (.x #^Vector3d other))
                        (- y (.y #^Vector3d other))
                        (- z (.z #^Vector3d other))))

  (mul [_ scalar] (let [scalar (double scalar)]
                    (Vector3d. (* x scalar) (* y scalar) (z scalar))))

  (div [_ scalar] (let [scalar (double scalar)]
                    (Vector3d. (* x scalar) (* y scalar) (z scalar))))

  (dist [_ other] (let [dx (double (- x (.x #^Vector3d other)))
                        dy (double (- y (.y #^Vector3d other)))
                        dz (double (- z (.z #^Vector3d other)))]
                    (Math/sqrt (prim double + (* dx dx) (* dy dy) (* dz dz)))))

  (dist-squared [_ other] (let [dx (double (- x (.x #^Vector3d other)))
                                dy (double (- y (.y #^Vector3d other)))
                                dz (double (- z (.z #^Vector3d other)))]
                            (prim double + (* dx dx) (* dy dy) (* dz dz))))

  (unit [_] (let [l (double (Math/sqrt (prim double + (* x x) (* y y) (* z z))))]
              (Vector3d. (/ x l) (/ y l) (/ z l))))

  (neg [_] (Vector3d. (* x (double -1.0)) (* y (double -1.0)) (* y (double -1.0))))
  
  (length [_] (Math/sqrt (prim double + (* x x) (* y y) (* z z))))

  (length-squared [_] (prim double + (* x x) (* y y) (* z z)))

  (dot [_ other] (prim double +
                       (* x (.x #^Vector3d other))
                       (* y (.y #^Vector3d other))
                       (* z (.z #^Vector3d other))))

  (cross [_ other] (let [ox (.x #^Vector3d other)
                         oy (.y #^Vector3d other)
                         oz (.z #^Vector3d other)]
                     (Vector3d. (prim double (- (* y oz) (* oy z)))
                                (prim double (- (* z ox) (* oz x)))
                                (prim double (- (* x oy) (* ox y)))))))

(defrecord Line3d [#^Vector3d direction #^Vector3d origin])

(defrecord Plane3d [#^float a #^float b #^float c #^float d #^Vector3d normal #^float D])

;; -----------------------------------------------------------------------------
;; Convenient values

(def zero (Vector3d. 0.0 0.0 0.0))
(def x-axis (Vector3d. 1.0 0.0 0.0))
(def y-axis (Vector3d. 0.0 1.0 0.0))
(def z-axis (Vector3d. 0.0 0.0 0.0))
